const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT

export const exportHeatsToCSV = async function(e, { heats }) {
  e.preventDefault()

  // csv header
  let csv = "created_at,name,location,planned_start\n"

  // append athlete data
  for (let i = 0; i < heats.length; i++) {
    csv += heats[i].created_at + "," + heats[i].name+ "," + heats[i].location+ ","
      + heats[i].planned_start+ "\n"
  }

  exportCSV(csv, "heats")
}

export const exportAthletesToCSV = async function(e, { athletes }) {
  e.preventDefault()

  // csv header
  let csv = "created_at,nr,firstname,lastname,birthday,school\n"

  // append athlete data
  for (let i = 0; i < athletes.length; i++) {
    csv += athletes[i].created_at + "," + athletes[i].nr + "," + athletes[i].firstname + ","
      + athletes[i].lastname + "," + athletes[i].birthday + "," + athletes[i].school + "\n"
  }

  exportCSV(csv, "athletes")
}

export const exportJudgesToCSV = async function(e, { judges }) {
  e.preventDefault()

  // csv header
  let csv = "created_at,email,firstname,lastname\n"

  // append judge data
  for (let i = 0; i < judges.length; i++) {
    csv += judges[i].created_at + "," + judges[i].email + ","
      + judges[i].firstname+ "," + judges[i].lastname + "\n"
  }

  exportCSV(csv, "judges")
}

export const exportLeaderboardToCSV = async function(e, leaderboard, heatSelection, rankingComp) {
  e.preventDefault()

  if (leaderboard.length === 0) {
    alert('Leaderboard is empty, nothing to export')
    return
  }

  // concatenate heat labels
  const heatNames = heatSelection.map(h => h.label)

  // rank leaderboard by selected comparator for each entry i in the board,
  // for every heat h
  const heatSummaries = leaderboard.sort(rankByHeat(rankingComp)).map(i =>
    heatSelection.map(h => formatScores(i, h))
  )

  // csv header
  let csv = "rank,start_nr,firstname,lastname,birthday,school,"
             + heatNames + "," + "best,worst,total\n"

  // append leaderboard score results
  for (let i = 0; i < leaderboard.length; i++) {
    csv += i+1 + "," + leaderboard[i].nr + "," + leaderboard[i].firstname + "," + leaderboard[i].lastname + ","
      + leaderboard[i].birthday + "," + leaderboard[i].school + ","
      + heatSummaries[i] + ","
      + leaderboard[i].bestHeat + "," + leaderboard[i].worstHeat + "," + leaderboard[i].sum + "\n"
  }

  exportCSV(csv, "scores")
}

// define the ranking logic
export const rankByHeat = function(rankingComp) {
  return function(a, b) {
    // rank by chosen heat or ranking comparator
    for (const r of rankingComp) {
      switch(r.value) {
        case 'start':
          // rank by start number
          return a.nr - b.nr
        case 'best':
          // rank by best heat first
          return b.bestHeat - a.bestHeat
          // rank by least worst heat for identical best heats
          return b.worstHeat - a.worstHeat
        case 'worst':
          // rank by worst heat
          return b.worstHeat - a.worstHeat
        case 'total':
          // rank by total sum across heats
          return b.sum - a.sum
        default:
          // rank by heat totals
          let aHeatTotal = a.heats.find(h => h.heatId === r.value)?.summary
          let bHeatTotal = b.heats.find(h => h.heatId === r.value)?.summary
          return bHeatTotal - aHeatTotal
      }
    }
  }
}

// Scores concat with "+" for leaderboard entry i and heat h
export const getScores = function(i, h) {
  const scores = i.heats.find(heat => heat.heatId === h.value)?.scores?.map(s => s.score.toFixed(1)).join(" + ")
  return scores
}
// Returns formatted string with summed scores and summary for leaderboard
// entry i and heat h
export const formatScores = function(i, h) {
  const scores = getScores(i, h)
  if (scores) {
    // get individual scores of the heat and score sum
    return getScores(i, h) + " = " + i.heats.find(heat => heat.heatId === h.value)?.summary.toFixed(1)
  } else {
    return NaN.toString()
  }
}

// export CSV as blob
const exportCSV = function(csv, fileSlug) {
  // create blob from csv
  const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' })

  // download blob
  const url = URL.createObjectURL(blob)
  const link = document.createElement('a')
  link.href = url
  link.download = fileSlug + "-" + new Date().toISOString() + ".csv"
  document.body.appendChild(link)
  link.click()
  document.body.removeChild(link)
  // let browser know not to keep the reference to the file
  URL.revokeObjectURL(url)
}
