import { lazy, useEffect, useState } from 'react'
import { useParams } from 'react-router-dom'
const Auth = lazy(() => import('./Auth'))
import { addAthleteToHeat } from './Leaderboard'
import Select from 'react-select'

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT
const locale = import.meta.env.VITE_LOCALE

async function addtoHeat(e, athlete, heatId, session) {
  e.preventDefault()
  try {
    await addAthleteToHeat(athlete.value, heatId, session)
    window.location.reload()
  } catch(error) {
    console.error(error)
  }
}

async function removeAthleteFromHeat(e, startlistId, athleteFirstName, athleteLastName, heatName, session) {
  e.preventDefault()
  const athleteName = athleteFirstName + (athleteLastName ? ' ' + athleteLastName : '')

  if (window.confirm('Do you really want to remove athlete "' + athleteName + '" from heat "' + heatName + '"?')) {
    const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/removeAthleteFromHeat`, {
      method: 'POST',
      headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${session.auth.token}`,
      },
      body: JSON.stringify({
        "startlist_id": startlistId,
      }),
    })
    const { data, error } = await res.json()
    if (error) {
      console.error(error)
    }
    window.location.reload()
  }
}

async function startlistWithAthletes(heatId, session) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/startlistWithAthletes`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "heat_id": heatId,
    }),
  })
  const { data, error } = await res.json()
  if (error) {
    throw error
  }
  return data
}

async function getHeat(heatId, session) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/getHeat`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "heat_id": heatId,
    }),
  })
  const { data, error } = await res.json()
  if (error) {
    throw error
  }
  return data
}

function StartlistForm({heatId, session}) {
  const [loading, setLoading] = useState(false)
  const [heatName, setheatName] = useState("")
  const [heatLocation, setheatLocation] = useState("")
  const [heatStart, setheatStart] = useState("")
  const [startlist, setStartlist] = useState([])
  const [athleteOpts, setAthleteOpts] = useState([])
  const [selectedAthlete, setselectedAthlete] = useState({})

  const dateOptions = {
            year: "numeric",
            month: "2-digit",
            day: "2-digit",
        }

  useEffect(() => {
    (async () => {
      setLoading(true)

      try {
        const startlist = await startlistWithAthletes(heatId, session)
        setStartlist(startlist)

        const heat = await getHeat(heatId, session)
        setheatName(heat.name)
        setheatLocation(heat.location)
        setheatStart(heat.planned_start)
      } catch (error) {
        console.error(error)
      }

      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allAthletes`, {
        headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${session.auth.token}`,
        },
      })
      const { data, error } = await res.json()
      if (error) {
        console.error(error)
      } else {
        let options = data.map(a => {
          return {
            value: a.id,
            label: a.nr + " " + a.firstname + " " + (a.lastname ? a.lastname : "")
          }
        })
        setAthleteOpts(options)
      }
      setLoading(false)
    })();
  }, [heatId])

  return (
    <div className='StartlistForm'>
      <button disabled={!loading} className='loading'>↺ loading</button>
      <h1>Startlist #{heatId} {heatName}</h1>
      <div className='heatInfo'>
        <ul>
          <li><b>Location:</b> {heatLocation ? heatLocation : 'n/a'}</li>
          <li><b>Planned start:</b> {heatStart ? heatStart : 'n/a'}</li>
        </ul>
      </div>
      <table>
        <thead>
          <tr>
            <th>Start Nr.</th>
            <th>Firstname</th>
            <th>Lastname</th>
            <th>Birthday</th>
            <th>School</th>
            <th>Add/Delete</th>
          </tr>
        </thead>
        <tbody>
        {startlist.map(i => (
            <tr key={i.startlist_id}>
              <td data-title='Start Nr.' className='right'>{i.nr}</td>
              <td data-title='Firstname'>{i.firstname}</td>
              <td data-title='Lastname'>{i.lastname}</td>
              <td data-title='Birthday'>{new Date(i.birthday).toLocaleDateString(locale, dateOptions)}</td>
              <td data-title='School'>{i.school}</td>
              <td><button onClick={e => removeAthleteFromHeat(
                e,
                i.startlist_id,
                i.firstname,
                i.lastname,
                heatName,
                session
              )}>&ndash; del</button></td>
            </tr>
          ))}
          <tr className='input'>
            <td data-title='Athlete' colSpan={5}>
              <Select
                options={athleteOpts}
                onChange={(e) => setselectedAthlete(e)}
              />
            </td>
            <td>
              <button onClick={(e) => addtoHeat(e, selectedAthlete, heatId, session)}>&#43; add</button>
            </td>
          </tr>
        </tbody>
      </table>
    </div>
  )
}

function Startlist({session}) {
    const { heatId } = useParams();

    return (
      <div>
        {!session.auth ? <Auth /> : <StartlistForm heatId={heatId} session={session} />}
      </div>
    )
  }
  
export default Startlist
