import { lazy, useEffect, useState } from 'react'
const Auth = lazy(() => import('./Auth'))

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT

async function updateSetting(e, session) {
  e.preventDefault()

  // Read the form data
  const formData = new FormData(e.target);
  const formJson = Object.fromEntries(formData.entries());
  console.log(formJson)

  // Add setting
  try {
    const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/updateSetting`, {
      method: 'POST',
      headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${session.auth.token}`,
      },
      body: JSON.stringify({
        "name": formJson.name === '' ? null : formJson.name,
        "value": formJson.value === '' ? null : formJson.value,
      }),
    })
    const { data, error } = await res.json()
    if (error) {
      alert('Failed to add setting: ' + error)
    }
    window.location.reload()
  } catch (error) {
    console.error('Failed to add setting: ' + error)
  }
}

async function deleteSetting(e, name, session) {
  e.preventDefault()

  if (window.confirm('Do you really want to delete setting "' + name + '"?')) {
    const { data, error } = await removeSetting(name, session)
    if (error === undefined) {
      window.location.reload()
    } else {
      console.error(error)
    }
  }
}

export async function removeSetting(name, session) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/removeSetting`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "name": name
    }),
  })
  return await res.json()
}


function SettingsForm({session}) {
  const [loading, setLoading] = useState(false)
  const [settings, setSettings] = useState([])

  useEffect(() => {
    (async () => {
      setLoading(true)
      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allSettings`, {
        headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${session.auth.token}`,
        }
      })
      if (res.status !== 204) {
        const { data, error } = await res.json()
        if (error) {
          console.error(error)
        } else {
          setSettings(data)
        }
      }
      setLoading(false)
    })();
  }, [])

  return (
    <div className='Settings'>
      <button disabled={!loading} className='loading'>↺ loading</button>
      <form method='post' onSubmit={e => updateSetting(e, session)}>
        <table>
          <thead>
            <tr>
              <th></th>
              <th>Setting *</th>
              <th>Value *</th>
              <th>New/delete</th>
            </tr>
          </thead>
          <tbody>
            {settings.map(s => (
              <tr key={s.name} data-name={s.name}>
                <td></td>
                <td data-title='Setting'>
                  {s.name}
                </td>
                <td data-title='Value'>
                  {s.value}
                </td>
                <td>
                  <button onClick={e => deleteSetting(e, s.name, session)}>&ndash; del</button>
                </td>
              </tr>
            ))}
            <tr className='input'>
              <td className='right'><i>* required</i></td>
              <td data-title='Setting *'>
                <input type='text' name='name' />
              </td>
              <td data-title='Value *'>
                <input type='text' name='value' />
              </td>
              <td>
                <button type='submit'>&#43; new</button>
              </td>
            </tr>
          </tbody>
          <tfoot>
            <tr>
              <td></td>
              <td></td>
              <td></td>
              <td>
              </td>
            </tr>
          </tfoot>
        </table>
      </form>
    </div>
  )
}

function Settings({session}) {
  return (
    <div>
      {!session.auth ? <Auth /> : <SettingsForm session={session} />}
    </div>
  )
}

export default Settings
