import { lazy, useEffect, useState } from 'react'
import { getStartlistForHeats } from './Leaderboard'
const Auth = lazy(() => import('./Auth'))
import Select from 'react-select'

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT

async function getScore(heatId, athleteId, session) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/getScore`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "heat": heatId,
      "athlete": athleteId,
      "judge": session.auth.id,
    }),
  })
  const { data, error } = await res.json()
  if (error) {
    throw error
  }
  return data
}

async function updateScore(score, heatId, athleteId, session) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/setScore`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "score": score,
      "heat": heatId,
      "athlete": athleteId,
      "judge": session.auth.id,
    }),
  })
  const { data, error } = await res.json()
  if (error) {
    throw error
  }
  return data
}

function ScoringForm({session}) {
  const [heats, setHeats] = useState([])
  const [heatSelection, setHeatSelection] = useState(0)
  const [athleteOpts, setAthleteOpts] = useState([])
  const [athleteSelection, setAthleteSelection] = useState(0)
  const [score, setScore] = useState(0)
  const [loading, setLoading] = useState(false)

  // add options to select or rank by heat
  const heatOpts = heats.map(h => {
    return {
      value: h.id,
      label: h.name
    }
  })

  useEffect(() => {
    (async () => {
      setLoading(true)
      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allHeats`)
      const { data, error } = await res.json()
      if (error) {
        console.error(error)
      }
      setHeats(data)

      let startlist = undefined
      try {
        startlist = await getStartlistForHeats([heatSelection.value])
      } catch (error) {
        console.error(error)
        setLoading(false)
        return
      }

      setAthleteOpts(startlist.map(s => {
        return {
          value: s.athlete,
          label: s.nr + " " + s.firstname + " " + (s.lastname ? s.lastname : "")
        }
      }))

      if (heatSelection.value === undefined || athleteSelection.value === undefined) {
        setLoading(false)
        return
      }

      // check if existing score for heat and athlete exists
      try {
        const currentScore = await getScore(
                heatSelection.value,
                athleteSelection.value,
                session
        )
        if (score === 0) {
          // fallback to current score when no new scoring took place
          setScore(currentScore.score)
        }
      } catch (error) {
        console.error(error)
      }

      // store new score
      try {
        await updateScore(score,
          heatSelection.value,
          athleteSelection.value,
          session)
      } catch (error) {
        console.error(error)
      }
      setLoading(false)
    })();
  }, [heatSelection, athleteSelection, session.auth.id, score]);

  return (
    <div className='Scoring'>
      <header>
        <button disabled={!loading} className='loading'>↺ loading</button>
      </header>
      <ul>
        <li>
          <label htmlFor='heat'>Heat</label>
          <Select
            options={heatOpts}
            onChange={h => { setHeatSelection(h); setScore(0) }}
            id='heat' />
        </li>
        <li>
          <label htmlFor='athlete'>Athlete</label>
          <Select
            options={athleteOpts}
            onChange={a => { setAthleteSelection(a); setScore(0) }}
            id='athlete' />
        </li>
        <li>
          <label htmlFor='score'>Score</label>
          <input
            className='scoreInput'
            type="number"
            value={score}
            onChange={(e) => setScore(e.target.value)}
            id='score' />
        </li>
      </ul>
    </div>
  )
}

function Score({session}) {
  return (
    <div>
      {!session.auth ? <Auth /> : <ScoringForm session={session} />}
    </div>
  )
}

export default Score
