import { exportLeaderboardToCSV, rankByHeat, formatScores } from './utils'
import { Fragment, useEffect, useState, useRef } from 'react'
import Select from 'react-select'
import { addNewHeat } from './Heats'

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT
const ws_uri = import.meta.env.VITE_WS_URI
const ws_port = import.meta.env.VITE_WS_PORT

const locale = import.meta.env.VITE_LOCALE

// use a socket for the real-time leaderboard data
let socket = new WebSocket(`${ws_uri}:${ws_port}/v1/leaderboard`);
console.info("Attached to server websocket");

socket.onclose = async function(event) {
  console.info("Server removed us from client list, reattaching socket");
  socket = new WebSocket(`${ws_uri}:${ws_port}/v1/leaderboard`);
}
socket.onopen = function (event) {
  // subscribe to scoring from judges when socket is opened
  socket.send(JSON.stringify({
    method: "watchScores",
  }))
}

export async function addAthleteToHeat(athlete, heat, session) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/addAthleteToHeat`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "athlete": athlete,
      "heat": heat,
    }),
  })
  const { data, error } = await res.json()
  if (error) {
    throw error
  }
  return data
}

export async function getStartlistForHeats(heatIds) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/distinctStartlist`, {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({
      "heat_ids": heatIds,
    }),
  })
  if (res.status === 204) {
    // return empty startlist
    return []
  } else {
    const { data, error } = await res.json()
    if (error) {
      throw error
    }
    return data
  }
}

async function getScoresForHeatAndAthlete(heatId, athleteId) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/scoresForHeatAndAthlete`, {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({
      "heat": heatId,
      "athlete": athleteId,
    }),
  })
  if (res.status !== 204) {
    const { data, error } = await res.json()
    if (error) {
      throw error
    }
    return data
  }
}

async function getScoreSummaryForHeatAndAthlete(heatId, athleteId) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/scoreSummaryForHeatAndAthlete`, {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({
      "heat": heatId,
      "athlete": athleteId,
    }),
  })
  if (res.status !== 204) {
    const { data, error } = await res.json()
    if (error) {
      throw error
    }
    return data
  }
}

async function getScoreSummary(heatIds) {
  const startListWithScores = []
  const startlist = await getStartlistForHeats(heatIds)

  for (const i of startlist) {
    i.heats = []

    for (const h of heatIds) {
      try {
        // this is an array, because the athlete can be scored by multiple judges
        const scores = await getScoresForHeatAndAthlete(h, i.athlete)
        const summary = await getScoreSummaryForHeatAndAthlete(h, i.athlete)
        if (scores && summary) {
          // add heat results of athlete to startlist entry
          i.heats.push({
            heatId: h,
            scores: scores,
            summary: summary.score_summary
          })
        }
      } catch (error) {
        console.error(error)
      }

      // find best/worst heat, use 'n/a' if no score yet
      i.bestHeat = i.heats.length > 0 ? Math.max(...i.heats.map(h => h.summary.toFixed(1))) : NaN.toString()
      i.worstHeat = i.heats.length > 0 ? Math.min(...i.heats.map(h => h.summary.toFixed(1))) : NaN.toString()

      // sum up all totals across heats
      i.sum = i.heats.map(h => h.summary).reduce((a, b) => a + b, 0).toFixed(1)
    }

    startListWithScores.push(i)
  }

  return startListWithScores
}

async function newHeatFromLeaderboard(e, {leaderboard, rankingComp, selectHeatRef, selectRankRef, session}) {
  e.preventDefault()

  if (leaderboard.length === 0) {
    alert("Cannot create new heat from empty leaderboard. Select heats to display.")
    return
  }

  // Read the form data
  const formData = new FormData(e.target);
  const formJson = Object.fromEntries(formData.entries());

  // create new heat
  let heat = undefined
  try {
    heat = await addNewHeat(
            formJson.name,
            formJson.location,
            formJson.planned_start === '' ? null : formJson.planned_start,
            session
    )
  } catch (error) {
    console.error(error)
  }

  const sortedBoard = leaderboard.sort(rankByHeat(rankingComp))
  for (let i = 0; i < formJson.size && i < sortedBoard.length; i++ ) {
    // add top N athletes from current leaderboard to new heat
    try {
      await addAthleteToHeat(sortedBoard[i].athlete, heat.id, session)
    } catch (error) {
      console.error(error)
    }
  }

  // clear values in selects to refresh list of heats
  selectHeatRef.current.clearValue()
  selectRankRef.current.clearValue()
  alert('Created new heat "' + formJson.name + '" with top ' + formJson.size + ' athletes')

  // todo: put heatOpts and rankOptions in state/useEffect again
  window.location.reload()
}

// export leaderboard with current ranking
function ExportForm({leaderboard, heatSelection, rankingComp}) {
  return (
    <div className='exportForm'>
      <form method='post' onSubmit={e => exportLeaderboardToCSV(
        e,
        leaderboard,
        heatSelection,
        rankingComp)}>
        <button type='submit'>&#9663; export</button>
      </form>
    </div>
  )
}

function NewHeatForm(leaderboard, rankingComp, selectHeatRef, selectRankRef, session) {
  return (
    <div className='newHeatForm'>
      <h2>New Heat from top N</h2>
      <p>
        Create new heat with top N athletes from the sorted leaderboard (<i>* required</i>).
      </p>
      <form method="post" onSubmit={(e) => newHeatFromLeaderboard(
        e,
        leaderboard,
        rankingComp,
        selectHeatRef,
        selectRankRef,
        session
      )}>
        <table>
          <thead>
            <tr>
              <th>New heat name *</th>
              <th>Location</th>
              <th>Planned start</th>
              <th>Include top N</th>
              <td></td>
            </tr>
          </thead>
          <tbody>
            <tr>
              <td data-title="New heat name *">
                <input type="text" name="name" />
              </td>
              <td data-title="Location">
                <input type="text" name="location" />
              </td>
              <td data-title="Planned start">
                <input type="time" name="planned_start" />
              </td>
              <td data-title="Include top N">
                <input type="number" name="size" />
              </td>
              <td>
                <button type="submit">&#43; new</button>
              </td>
            </tr>
          </tbody>
        </table>
      </form>
    </div>
  )
}

function Leaderboard({session}) {
  const [loading, setLoading] = useState(false)
  const [leaderboard, setLeaderboard] = useState([])
  const [heatSelection, setHeatSelection] = useState([])
  const [heats, setHeats] = useState([])
  const [rankingComp, setRankingComp] = useState([])
  const [details, showDetails] = useState(false)

  const selectHeatRef = useRef();
  const selectRankRef = useRef();

  const dateOptions = {
            year: "numeric",
            month: "2-digit",
            day: "2-digit",
        }

  // add options to select or rank by heat
  const heatOpts = heats.map(h => {
    return {
      value: h.id,
      label: h.name
    }
  })

  // add static ranking options
  const rankOptions = [
    {
      value: 'start',
      label: 'Start Nr.'
    }, {
      value: 'best',
      label: 'Best Heat'
    }, {
      value: 'worst',
      label: 'Worst Heat'
    }, {
      value: 'total',
      label: 'Total Sum (all heats)'
    }
  ]

  // add dynamic options to rank by best/worst heat
  const heatOptions = heatOpts.map(h => {
    return {
      value: h.value,
      label: "Sum " + h.label
    }
  })

  const rankOpts = [
    {
      label: "Overall",
      options: rankOptions
    },
    {
      label: "Heat Sum",
      options: heatOptions
    }
  ]

  useEffect(() => {
    (async () => {
      setLoading(true)

      // load initial list of heats
      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allHeats`)
      const { data, error } = await res.json()
      if (error) {
        console.error(error)
      }
      setHeats(data)

      // default to rank by total
      setRankingComp([rankOptions[3]])
      setLoading(false)
    })();
  }, []);

  useEffect(() => {
    (async () => {
      setLoading(true)

      // reload entire leaderboard when heat selection is changed
      const scoreSummary = await getScoreSummary(heatSelection.map(h => h.value))
      setLeaderboard(scoreSummary)
      setLoading(false)
    })();
  }, [heatSelection]);

  useEffect(() => {
    (async() => {
      socket.onmessage = async function(event) {
        // todo: reload only required scores
        const scoreSummary = await getScoreSummary(heatSelection.map(h => h.value))
        setLeaderboard(scoreSummary)
      }
      setLoading(false)
    })();
  }, [heatSelection]);

  return (
    <>
      <div className="Leaderboard">
        <header>
          <button disabled={!loading} className="loading">
            ↺ loading
          </button>
          <button className={`show-details ${details ? "toggled" : ""}`} onClick={() => showDetails(!details)}>
            <div className="thumb"></div>
            <span>{details ? "less" : "more"}</span>
          </button>
          <label htmlFor="heat">Heats to display</label>
          <Select
            closeMenuOnSelect={false}
            isMulti
            options={heatOpts}
            onChange={(h) => setHeatSelection(h)}
            ref={selectHeatRef}
            id="heat"
          />
          <label htmlFor="rank" className={details ? "" : "hidden"}>
            Rank by
          </label>
          <Select
            closeMenuOnSelect={false}
            isMulti
            options={rankOpts}
            defaultValue={rankOpts[0].options[3]}
            onChange={(h) => setRankingComp(h)}
            ref={selectRankRef}
            className={details ? "" : "hidden"}
            id="rank"
          />
        </header>
        <table className={details ? "leaderboard" : "hide-rank"}>
          <thead>
            <tr>
              <th className={details ? "right" : "hidden"}>Rank</th>
              <th className="right">Start Nr.</th>
              <th>Name</th>
              <th className={details ? "" : "hidden"}>Birthday</th>
              <th className={details ? "" : "hidden"}>School</th>
              {heatSelection.map((h) => (
                <th className={details ? "right" : "hidden"} key={h.value}>
                  {h.label}
                </th>
              ))}
              <th className={details ? "right" : "hidden"}>Best</th>
              <th className={details ? "right" : "hidden"}>Worst</th>
              <th className="right">Total</th>
            </tr>
          </thead>
          <tbody>
            {leaderboard.sort(rankByHeat(rankingComp)).map((i) => (
              <tr key={i.id}>
                <td className={details ? "right" : "hidden"}></td>
                <td data-title="Start Nr." className="right">
                  {i.nr}
                </td>
                <td data-title="Name">
                  {i.firstname} {i.lastname}
                </td>
                <td data-title="Birthday" className={details ? "" : "hidden"}>
                  {i.birthday ? new Date(i.birthday).toLocaleDateString(
                        locale, dateOptions) : ""}
                </td>
                <td data-title="School" className={details ? "" : "hidden"}>
                  {i.school}
                </td>
                {heatSelection.map((h) => (
                  // list all scores from the judges seperated with '+' signs, show sum on right side
                  <td key={h.value} className={details ? "right" : "hidden"} data-title={h.label}>
                    {formatScores(i, h)}
                  </td>
                ))}
                <td className={details ? "right" : "hidden"} data-title="Best">
                  {i.bestHeat}
                </td>
                <td className={details ? "right" : "hidden"} data-title="Worst">
                  {i.worstHeat}
                </td>
                <td className="right total" data-title="Total">
                  {i.sum}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <ExportForm
        leaderboard={leaderboard}
        heatSelection={heatSelection}
        rankingComp={rankingComp}
      />
      {session.auth ? (
        <NewHeatForm
          leaderboard={leaderboard}
          rankingComp={rankingComp}
          selectHeatRef={selectHeatRef}
          selectRankRef={selectRankRef}
          session={session}
        />) : ("")}
    </>
  )
}

export default Leaderboard
