import { lazy, useEffect, useState } from 'react'
import { exportJudgesToCSV } from './utils'
const Auth = lazy(() => import('./Auth'))

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT
const locale = import.meta.env.VITE_LOCALE

async function addJudge(e, session) {
  e.preventDefault()

  // Read the form data
  const formData = new FormData(e.target);
  const formJson = Object.fromEntries(formData.entries());

  // create new judge
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/addJudge`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "email": formJson.email,
      "firstname": formJson.firstname,
      "lastname": formJson.lastname
    }),
  })
  const { data, error } = await res.json()
  if (error) {
    alert('Failed to create new judge: ' + error.message)
  }
  window.location.reload()
}

async function deleteJudge(e, judgeId, email, session) {
  e.preventDefault()

  if (window.confirm(`Do you really want to delete judge ${email}?`)) {
    const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/removeJudge`, {
      method: 'POST',
      headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${session.auth.token}`,
      },
      body: JSON.stringify({
        "judge_id": judgeId,
      }),
    })
    const { data, error } = await res.json()
    if (error) {
      alert('Failed to delete judge: ' + error.message)
    }
    window.location.reload()
  }
}

// export judges
function ExportForm(judges) {
  return (
    <div className='exportForm'>
      <form method='post' onSubmit={e => exportJudgesToCSV(e, judges)}>
        <button type='submit'>&#9663; export</button>
      </form>
    </div>
  )
}

function JudgeForm({session}) {
  const [loading, setLoading] = useState(false)
  const [judges, setJudges] = useState([])
  const dateOptions = {
            year: "numeric",
            month: "2-digit",
            day: "2-digit",
        }

  useEffect(() => {
    (async () => {
      setLoading(true)
      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allJudges`, {
        headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${session.auth.token}`,
        }
      })
      const { data, error } = await res.json()
      if (error) {
        console.log(error)
      } else {
        setJudges(data)
      }
      setLoading(false)
    })();
  }, [])

  return (
    <div className='JudgeForm'>
      <button disabled={!loading} className='loading'>↺ loading</button>
      <form method='post' onSubmit={e => addJudge(e, session)}>
        <table>
          <thead>
            <tr>
              <th>Created at</th>
              <th>Email *</th>
              <th>Firstname</th>
              <th>Lastname</th>
            </tr>
          </thead>
          <tbody>
            {judges.map(j => (
              <tr key={j.id}>
                <td data-title='Created at' className='right'>{new Date(j.created_at).toLocaleDateString(locale, dateOptions)}</td>
                <td data-title='Email'>{j.email}</td>
                <td data-title='Firstname'>{j.firstname}</td>
                <td data-title='Lastname'>{j.lastname}</td>
                <td><button onClick={e => deleteJudge(e, j.id, j.email, session)}>&ndash; del</button></td>
              </tr>
            ))}
            <tr className='input'>
              <td className='right'><i>* required</i></td>
              <td data-title='Email *'>
                <input type='text' name='email' />
              </td>
              <td data-title='Firstname'>
                <input type='text' name='firstname' />
              </td>
              <td data-title='Lastname'>
                <input type='text' name='lastname' />
              </td>
              <td>
                <button type='submit'>&#43; new</button>
              </td>
            </tr>
          </tbody>
        </table>
      </form>
      <ExportForm judges={judges} />
    </div>
  )
}

function Judges({session}) {
    return (
      <div>
        {!session.auth ? <Auth /> : <JudgeForm session={session} />}
      </div>
    )
  }
  
export default Judges
