import { lazy, useEffect, useState } from 'react'
import { generatePath, Link } from 'react-router-dom'
import { exportHeatsToCSV } from './utils'

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT
const locale = import.meta.env.VITE_LOCALE

const Auth = lazy(() => import('./Auth'))

export async function addNewHeat(name, heatLocation, plannedStart, session) {
  try {
    const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/newHeat`, {
      method: 'POST',
      headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${session.auth.token}`,
      },
      body: JSON.stringify({
        "name": name,
        "location": heatLocation,
        "planned_start": plannedStart
      }),
    })
    const { data, error } = await res.json()
    return data
  } catch (error) {
    throw(error)
  }
}

async function addHeat(e, session) {
  e.preventDefault()

  // Read the form data
  const formData = new FormData(e.target);
  const formJson = Object.fromEntries(formData.entries());

  // create new heat
  try {
    const heat = await addNewHeat(
            formJson.name,
            formJson.location,
            // planned_start is an empty string if unset
            // https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input/time
            formJson.planned_start === '' ? null : formJson.planned_start,
	    session
    )
    window.location.reload()
  } catch (error) {
    console.error('Failed to create new heat: ' + error.message)
  }
}

export async function removeHeat(heatId, session) {
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/removeHeat`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "heat_id": heatId
    }),
  })
  return await res.json()
}

async function deleteHeat(e, heatId, heatName, session) {
  e.preventDefault()

  if (window.confirm('Do you really want to delete heat "' + heatName + '"?')) {
    const { data, error } = await removeHeat(heatId, session)
    if (error === undefined) {
      window.location.reload()
    } else {
      console.error(error)
    }
  }
}

// export heats
function ExportForm(heats) {
  return (
    <div className='exportForm'>
      <form method='post' onSubmit={e => exportHeatsToCSV(e, heats)}>
        <button type='submit'>&#9663; export</button>
      </form>
    </div>
  )
}

function HeatForm({session}) {
  const [loading, setLoading] = useState(false)
  const [heats, setHeats] = useState([])
  const dateOptions = {
            year: "numeric",
            month: "2-digit",
            day: "2-digit",
        }

  useEffect(() => {
    (async () => {
      setLoading(true)
      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allHeats`)
      const { data, error } = await res.json()
      if (error) {
        console.error(error)
      }
      setHeats(data)
      setLoading(false)
    })();
  }, [])

  return (
    <div className='HeatForm'>
      <button disabled={!loading} className='loading'>↺ loading</button>
      <form method='post' onSubmit={e => addHeat(e, session)}>
        <table>
          <thead>
            <tr>
              <th>Created at</th>
              <th>Name *</th>
              <th>Location</th>
              <th>Planned start</th>
              <th>New/delete</th>
            </tr>
          </thead>
          <tbody>
            {heats.map(h => (
              <tr key={h.id}>
                <td data-title='Created at' className='right'>{new Date(h.created_at).toLocaleDateString(locale, dateOptions)}</td>
                <td data-title='Name'><Link to={generatePath('/heats/startlist/:heatId', {heatId:h.id})}>{h.name}</Link></td>
                <td data-title='Location'>{h.location}</td>
                <td data-title='Planned start'>{h.planned_start}</td>
                <td><button onClick={e => deleteHeat(e, h.id, h.name, session)}>&ndash; del</button></td>
              </tr>
            ))}
            <tr className='input'>
              <td className='right'><i>* required</i></td>
              <td data-title='Name *'>
                <input type='text' name='name' />
              </td>
              <td data-title='Location'>
                <input type='text' name='location' />
              </td>
              <td data-title='Planned start' className='right'>
                <input
                  type='time'
                  name='planned_start' />
              </td>
              <td>
                <button type='submit'>&#43; new</button>
              </td>
            </tr>
          </tbody>
        </table>
      </form>
      <ExportForm heats={heats} />
    </div>
  )
}

function Heats({session}) {
    return (
      <div>
        {!session.auth ? <Auth /> : <HeatForm session={session} />}
      </div>
    )
  }
  
export default Heats;
