import { lazy, useEffect, useState } from 'react'
import { exportAthletesToCSV } from './utils'
const Auth = lazy(() => import('./Auth'))

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT
const locale = import.meta.env.VITE_LOCALE

async function addAthlete(e, session) {
  e.preventDefault()

  // Read the form data
  const formData = new FormData(e.target);
  const formJson = Object.fromEntries(formData.entries());

  // create new athlete
  const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/addAthlete`, {
    method: 'POST',
    headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.auth.token}`,
    },
    body: JSON.stringify({
      "nr": formJson.nr,
      "firstname": formJson.firstname,
      "lastname": formJson.lastname,
      "birthday": formJson.birthday ? formJson.birthday : null,
      "school": formJson.school
    }),
  })
  const { data, error } = await res.json()
  if (error) {
    alert('Failed to create new athlete: ' + error.message)
  }
  window.location.reload()
}

async function deleteAthlete(e, athleteId, athleteFirstName, athleteLastName, session) {
  e.preventDefault()

  const athleteName = athleteFirstName + (athleteLastName ? ' ' + athleteLastName : '')
  if (window.confirm('Do you really want to delete athlete "' + athleteName + '"?')) {
    const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/removeAthlete`, {
      method: 'POST',
      headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${session.auth.token}`,
      },
      body: JSON.stringify({
        "athlete_id": athleteId,
      }),
    })
    const { data, error } = await res.json()
    if (error) {
      alert('Failed to delete athlete: ' + error.message)
    }
    window.location.reload()
  }
}

// export athletes
function ExportForm(athletes) {
  return (
    <div className='exportForm'>
      <form method='post' onSubmit={e => exportAthletesToCSV(e, athletes)}>
        <button type='submit'>&#9663; export</button>
      </form>
    </div>
  )
}

function AthleteForm({session}) {
  const [loading, setLoading] = useState(false)
  const [athletes, setAthletes] = useState([])
  const dateOptions = {
            year: "numeric",
            month: "2-digit",
            day: "2-digit",
        }

  useEffect(() => {
    (async () => {
      setLoading(true)
      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allAthletes`, {
        headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${session.auth.token}`,
        }
      })
      const { data, error } = await res.json()
      if (error) {
        console.log(error)
      } else {
        setAthletes(data)
      }
      setLoading(false)
    })();
  }, [])

  return (
    <div className='AthleteForm'>
      <button disabled={!loading} className='loading'>↺ loading</button>
      <form method='post' onSubmit={e => addAthlete(e, session)}>
        <table>
          <thead>
            <tr>
              <th>Created at</th>
              <th className="right">Start Nr. *</th>
              <th>Firstname *</th>
              <th>Lastname</th>
              <th>Birthday</th>
              <th>School</th>
              <th>New/delete</th>
            </tr>
          </thead>
          <tbody>
            {athletes.map(a => (
              <tr key={a.id}>
                <td data-title='Created at' className='right'>{new Date(a.created_at).toLocaleDateString(locale, dateOptions)}</td>
                <td data-title='Start Nr.' className='right'>{a.nr}</td>
                <td data-title='Firstname'>{a.firstname}</td>
                <td data-title='Lastname'>{a.lastname}</td>
                <td data-title='Birthday'>{a.birthday ? new Date(a.birthday).toLocaleDateString(locale, dateOptions) : ''}</td>
                <td data-title='School'>{a.school}</td>
                <td><button onClick={e => deleteAthlete(e, a.id, a.firstname, a.lastname, session)}>&ndash; del</button></td>
              </tr>
            ))}
            <tr className='input'>
              <td className='right'><i>* required</i></td>
              <td data-title='Start Nr. *' className='right'>
                <input type='number' name='nr' />
              </td>
              <td data-title='Firstname *'>
                <input type='text' name='firstname' />
              </td>
              <td data-title='Lastname'>
                <input type='text' name='lastname' />
              </td>
              <td data-title='Birthday' className='right'>
                <input
                  type='date'
                  name='birthday' />
              </td>
              <td data-title='School'>
                <input type='text' name='school' />
              </td>
              <td>
                <button type='submit'>&#43; new</button>
              </td>
            </tr>
          </tbody>
        </table>
      </form>
      <ExportForm athletes={athletes} />
    </div>
  )
}

function Athletes({session}) {
    return (
      <div>
        {!session.auth ? <Auth /> : <AthleteForm session={session} />}
      </div>
    )
  }
  
export default Athletes
