import { Suspense, lazy, useState, useEffect, Fragment } from 'react'
import { BrowserRouter as Router, Routes, Route, Outlet, Link, NavLink } from 'react-router-dom'
import { CookiesProvider, useCookies } from 'react-cookie'

import './css/App.css'
import './css/blue.css'
import './css/red.css'
import './css/yellow.css'

const Score = lazy(() => import('./Score'))
const Heats = lazy(() => import('./Heats'))
const Judges = lazy(() => import('./Judges'))
const Athletes = lazy(() => import('./Athletes'))
const Startlist = lazy(() => import('./Startlist'))
const Auth = lazy(() => import('./Auth'))
const AuthVerify = lazy(() => import('./AuthVerify'))
const Leaderboard = lazy(() => import('./Leaderboard'))
const Settings = lazy(() => import('./Settings'))

const api_uri = import.meta.env.VITE_API_URI
const api_port = import.meta.env.VITE_API_PORT

document.title = import.meta.env.VITE_APP_DOC_TITLE ? import.meta.env.VITE_APP_DOC_TITLE : 'My Heats'

function Layout() {
  const [session, setSession, destroySession] = useCookies(['auth'])
  const [settings, setSettings] = useState(new Map())

  // load stylesheet based on settings
  const colors = ['red', 'blue', 'yellow'];
  let theme = ""
  let showLogo = "true"
  let logoUrl = "/logo192.png"
  let showVersion = "true"

  if (settings.style && colors.includes(settings.style)) {
    theme = settings.style
    //console.info(`I see ${settings.style}. Try one of "style=${colors}" in ${window.location.origin}/settings`)
  }
  if (settings.showLogo) {
    showLogo = settings.showLogo
  }
  if (settings.logoUrl) {
    logoUrl = settings.logoUrl
  }
  if (settings.showVersion) {
    showVersion = settings.showVersion
  }

  useEffect(() => {
    (async () => {
      const res = await fetch(`${api_uri}:${api_port}/v1/leaderboard/allSettings`)
      if (res.status !== 204) {
        const { data, error } = await res.json()
        if (error) {
          console.error(error)
        } else {
          let s = new Map()
          data.forEach(setting => {
                  s[setting.name] = setting.value
          })
          setSettings(s)
        }
      }
    })();
  }, [])

  return (
    <>
      <nav className={theme}>
        <ul>
          <li>
            <NavLink
              className={({ isActive, isPending }) => isPending ? "pending" : isActive ? `active ${theme}.active` : ""}
              to="/">
                Leaderboard
            </NavLink>
          </li>
          {session.auth ? <li>
            <NavLink
              className={({ isActive, isPending }) => isPending ? "pending" : isActive ? `active ${theme}.active}` : ""}
              to="/score">
                Scoring
            </NavLink>
          </li> : ''}
          {session.auth ? <li>
            <NavLink
              className={({ isActive, isPending }) => isPending ? "pending" : isActive ? `active ${theme}.active}` : ""}
              to="/heats">
                Heats and Startlists
            </NavLink>
          </li> : ''}
          {session.auth ? <li>
            <NavLink
              className={({ isActive, isPending }) => isPending ? "pending" : isActive ? `active ${theme}.active}` : ""}
              to="/athletes">
                Athletes
            </NavLink>
          </li> : ''}
          {session.auth ? <li>
            <NavLink
              className={({ isActive, isPending }) => isPending ? "pending" : isActive ? `active ${theme}.active}` : ""}
              to="/judges">
                Judges
            </NavLink>
          </li> : ''}
        </ul>
      </nav>
      <main>
        <Outlet />
      </main>
      <footer>
        <br />
        {showVersion === 'true' ?
          <span>MyHeats <a href="https://code.in0rdr.ch/myheats/refs.html">v0.8</a></span> : '' }
        {showLogo === 'true' ?
          <span className="logo"><img src={logoUrl} alt="Logo" /></span> : '' }
        <span>
          {session.auth ? <button
              onClick={() => location.href="/settings"}>
                &#9881;
            </button> : ''}
        </span>
        <span>
          {session.auth ? <button onClick={() => destroySession('auth')}>&#10157;</button> :
            <NavLink
              className={({ isActive, isPending }) => isPending ? "pending" : isActive ? "active" : ""}
              to="/auth">
                Login
            </NavLink>
          }
        </span>
      </footer>
    </>
  )
}

function NoMatch() {
  return (
    <div className="NoMatch">
      Nothing to see here, <Link to="/">go to leaderboard</Link>
    </div>
  )
}

function App() {
  const [session] = useCookies(['auth'])

  return (
    <Fragment>
      <Router>
        <Suspense fallback={<div>Loading...</div>}>
          <Routes>
            <Route path="/" element={<Layout session={session} />}>
              <Route path="/" element={<Leaderboard session={session} />} />
              <Route path="/score" element={<Score session={session} />} />
              <Route path="/heats" element={<Heats session={session} />} />
              <Route path="/judges" element={<Judges session={session} />} />
              <Route path="/athletes" element={<Athletes session={session} />} />
              <Route path="/heats/startlist/:heatId"  element={<Startlist session={session} />} />
              <Route path="/auth" element={<Auth />} />
              <Route path="/authverify" element={<AuthVerify />} />
              <Route path="/settings" element={<Settings session={session} />} />
              <Route path="*" element={<NoMatch />} />
            </Route>
          </Routes>
        </Suspense>
      </Router>
    </Fragment>
  );
}

export default App;
