const ws = require('ws');
const http = require('node:http');
const url = require('url');
const jwt = require('jsonwebtoken');
const nodemailer = require('nodemailer');
const db = require('./db.cjs');

require('dotenv').config({
  // Configure common config files and modes
  // - https://www.npmjs.com/package/dotenv
  // - https://vitejs.dev/guide/env-and-mode
  path: [
    '.env.local',
    '.env.development', '.env.development.local',
    '.env.test',        '.env.test.local',
    '.env.production',  '.env.production.local',
    '.env'
  ]
});

// Read environment and set defaults
api_uri = process.env.API_URI;
api_port = process.env.API_PORT;
redirect_uri = process.env.API_REDIRECT_URI;
cors_allow_origin = process.env.API_CORS_ALLOW_ORIGIN;
jwt_secret = process.env.API_JWT_SECRET;
jwt_ttl = process.env.API_JWT_TTL;

if (jwt_secret === undefined) {
  console.log('Balls, API_JWT_SECRET undefined 🍳');
  process.exit()
} else if (process.env.SMTP_HOST === undefined ||
           process.env.SMTP_PORT === undefined ||
           process.env.SMTP_STARTTLS === undefined ||
           process.env.SMTP_USER === undefined ||
           process.env.SMTP_FROM === undefined ||
           process.env.SMTP_PASSWORD === undefined) {
  console.log('Balls, Not all SMTP settings defined 🍳');
  process.exit()
}

// Configure mail transport
// https://nodemailer.com/smtp
const transport = nodemailer.createTransport({
  host: process.env.SMTP_HOST,
  port: process.env.SMTP_PORT,
  secure: process.env.SMTP_STARTTLS,
  auth: {
    user: process.env.SMTP_USER,
    pass: process.env.SMTP_PASSWORD,
  },
});

// Define API paths and allowed request methods
const paths = [
    '/v1/healthz',
    '/v1/auth/verify',
    '/v1/echo',
    '/v1/auth/requestMagicLink',
    '/v1/auth/invalidateToken',                      // not implemented
    '/v1/leaderboard/allHeats',
    '/v1/leaderboard/allJudges',                     // 🔒 authenticated
    '/v1/leaderboard/allAthletes',                   // 🔒 authenticated
    '/v1/leaderboard/newHeat',                       // 🔒 authenticated
    '/v1/leaderboard/getHeat',                       // 🔒 authenticated
    '/v1/leaderboard/removeHeat',                    // 🔒 authenticated
    '/v1/leaderboard/distinctStartlist',
    '/v1/leaderboard/startlistWithAthletes',         // 🔒 authenticated
    '/v1/leaderboard/scoresForHeatAndAthlete',
    '/v1/leaderboard/scoreSummaryForHeatAndAthlete',
    '/v1/leaderboard/getScore',                      // 🔒 authenticated
    '/v1/leaderboard/setScore',                      // 🔒 authenticated
    '/v1/leaderboard/addAthleteToHeat',              // 🔒 authenticated
    '/v1/leaderboard/removeAthleteFromHeat',         // 🔒 authenticated
    '/v1/leaderboard/addAthlete',                    // 🔒 authenticated
    '/v1/leaderboard/removeAthlete',                 // 🔒 authenticated
    '/v1/leaderboard/addJudge',                      // 🔒 authenticated
    '/v1/leaderboard/removeJudge',                   // 🔒 authenticated
    '/v1/leaderboard/allSettings',
    '/v1/leaderboard/getSetting',                    // 🔒 authenticated
    '/v1/leaderboard/updateSetting',                 // 🔒 authenticated
    '/v1/leaderboard/removeSetting',                 // 🔒 authenticated
]

console.log("Backend API:", api_uri);
console.log("Backend API port:", api_port);
console.log("Redirect URI:", redirect_uri);
console.log("CORS Allow Origin:", cors_allow_origin);
console.log("SMTP server:", process.env.SMTP_HOST);
console.log("SMTP port:", process.env.SMTP_PORT);
console.log("SMTP starttls:", process.env.SMTP_STARTTLS);
console.log("* * * * * * *", "Uncle roger ready 🍚", "* * * * * * *");

// Create a simple http API server
// - https://nodejs.org/en/learn/modules/anatomy-of-an-http-transaction
// - https://nodejs.org/api/http.html#httpcreateserveroptions-requestlistener
const server = http.createServer();

// Listen on request events
server.on('request', async (req, res) => {
  const url = new URL(`${req.protocol}://${req.host}${req.url}`);
  const search_params = url.searchParams;

  console.log('> Path', url.pathname);
  console.log('  Method', req.method);

  // set headers
  res.setHeader('Content-Type', 'application/json');
  res.setHeader('Access-Control-Allow-Origin', cors_allow_origin);

  // cors pre-flight request uses options method
  res.setHeader('Access-Control-Allow-Methods', 'GET, POST, OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type,Authorization');

  // extact authorization token
  const authHeader = req.headers.authorization
  let token = undefined
  try {
    if (authHeader.startsWith("Bearer ")) {
      token = authHeader.substring(7, authHeader.length)
      console.log('  Bearer token', token);
    }
  } catch(error) {
    console.warn("x Warning: No authorization bearer token in request header")
    //serverError(res, new Error("No authorization bearer token in request header"));
    //return
  }

  // cors pre-flight request uses options method
  if (req.method === 'OPTIONS') {
    res.end()
  } else if (req.method === 'GET') {
    switch(url.pathname) {
      case '/v1/healthz':
        res.end(JSON.stringify({
          message: 'egg fried rice 🍚 fuiyooh!',
        }));
        break
      case '/v1/auth/verify':
        try {
          token = search_params.get('token');
          let user = await verifyToken(req, token)
          res.end(JSON.stringify({
            message: 'Uncle roger verified 🤝 fuuiiyooh!',
            data: user,
          }));
        } catch (error) {
          serverError(res, error);
        }
        break
      case '/v1/leaderboard/allHeats':
        try {
          const heats = await db.allHeats()

          res.end(JSON.stringify({
            message: 'All heats',
            data: heats,
          }));
        } catch(error) {
          serverError(res, error);
        }
        break
      case '/v1/leaderboard/allJudges':
        try {
          await verifyToken(req, token)
          const judges = await db.allJudges()

          res.end(JSON.stringify({
            message: 'All judges',
            data: judges,
          }));
        } catch(error) {
          serverError(res, error);
        }
        break
      case '/v1/leaderboard/allAthletes':
        try {
          await verifyToken(req, token)
          const athletes = await db.allAthletes()

          res.end(JSON.stringify({
            message: 'All athletes',
            data: athletes,
          }));
        } catch(error) {
          serverError(res, error);
        }
        break
      case '/v1/leaderboard/getSetting':
        try {
          await verifyToken(req, token)

          let name = search_params.get('name');
          const setting = await db.getSetting(name)

          if (setting.length < 1) {
            noContent(res);
            return
          }
          res.end(JSON.stringify({
            message: `Setting with name ${name}`,
            data: setting[0].value,
          }));
        } catch(error) {
          serverError(res, error);
        }
        break
      case '/v1/leaderboard/allSettings':
        try {
          const settings = await db.allSettings()

          if (settings.length < 1) {
            noContent(res);
            return
          }
          res.end(JSON.stringify({
            message: 'All settings',
            data: settings,
          }));
        } catch(error) {
          serverError(res, error);
        }
        break
      default:
        const pathExists = paths.find((i) => i === url.pathname);
        if (pathExists) {
          // wrong method for this path
          notAllowed(res, req.method);
        } else {
          notFound(res, url.pathname);
        }
    } // end switch
  } else if (req.method === 'POST') {
    let body = [];

    switch(url.pathname) {
      case '/v1/echo':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', () => {
          body = JSON.stringify({
            message: 'Uncle roger knows do egg fried rice since 3 years old, lah',
            data: Buffer.concat(body).toString(),
          });
          res.end(body);
        })
        break
      case '/v1/auth/requestMagicLink':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  Link request for:', input.email);

            const users = await db.getUser(input.email);
            if (users.length < 1) {
              throw new Error("User not found")
            }

            const user = users[0]

            // create magic link
            const exp = new Date(Date.now() + 1000 * jwt_ttl)
            const token = jwt.sign({ email: user.email }, jwt_secret, {
              expiresIn: 1000 * jwt_ttl,
            })
            const magic_link = `${redirect_uri}?token=${token}`
            const magic_href = `${api_uri}:${api_port}/v1/auth/verify?token=${token}`
            console.log("🔗Magic href:", magic_href)

            // save token with expiration time
            await db.saveToken(user.id, token, exp)

            // send magic link to user
            await transport.sendMail({
              from: process.env.SMTP_FROM,
              to: user.email,
              subject: 'MyHeats Magic Link',
              //text: `Click here ✨ to log in: ${magic_link}`,
              html: `Click <a href="${magic_link}">here</a> ✨ to log in.`,
            })

            res.end(JSON.stringify({
              message: 'Magic link requested',
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/distinctStartlist':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  distinctStartlist request with headIds:', input.heat_ids);

            const startlist = await db.distinctStartlist(input.heat_ids);

            if (startlist.length < 1) {
              noContent(res);
              return
            }
            res.end(JSON.stringify({
              message: 'Distinct athletes for multiple heats',
              data: startlist,
            }));
          } catch(error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/startlistWithAthletes':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  startlistWithAthletes request with headId:', input.heat_id);

            const startlist = await db.startlistWithAthletes(input.heat_id);

            if (startlist.length < 1) {
              throw new Error("No athletes for this startlist")
            }
            res.end(JSON.stringify({
              message: 'Startlist with athletes for heat',
              data: startlist,
            }));
          } catch(error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/scoresForHeatAndAthlete':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  scoresForHeatAndAthlete request with heat and athlete:',
                    input.heat, input.athlete);

            const scores = await db.scoresForHeatAndAthlete(
                    input.heat,
                    input.athlete
            )

            if (scores.length < 1) {
              noContent(res);
              return
            }
            res.end(JSON.stringify({
              message: 'Scores for heat and athlete',
              data: scores,
            }));
          } catch(error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/scoreSummaryForHeatAndAthlete':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  scoreSummaryForHeatAndAthlete request with heat and athlete:',
                    input.heat, input.athlete);

            const summary = await db.scoreSummaryForHeatAndAthlete(
                    input.heat,
                    input.athlete
            )

            if (summary.length < 1) {
              noContent(res);
              return
            }
            res.end(JSON.stringify({
              message: 'Score summary for heat and athlete',
              data: summary[0],
            }));
          } catch(error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/getScore':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  GetScore request for:', input);

            const scores = await db.getScore(
                    input.heat,
                    input.athlete,
                    input.judge
            );

            if (scores.length < 1) {
              noContent(res);
              return
            }

            res.end(JSON.stringify({
              message: 'Requested score for heat, user and judge',
              data: scores[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/setScore':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  SetScore request for:', input);

            const scores = await db.setScore(
                    input.heat,
                    input.athlete,
                    input.judge,
                    input.score,
            );

            if (scores.length < 1) {
              throw new Error("Score not updated")
            }

            res.end(JSON.stringify({
              message: 'Score update for heat, user and judge',
              data: scores[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/newHeat':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  newHeat request for:', input);

            const heats = await db.newHeat(
                    input.name,
                    input.location,
                    input.planned_start,
            );
            if (heats.length < 1) {
              throw new Error("Heat not created")
            }

            res.end(JSON.stringify({
              message: 'New heat',
              data: heats[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/getHeat':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  getHeat request for:', input);

            const heats = await db.getHeat(
                    input.heat_id,
            );
            if (heats.length < 1) {
              throw new Error("Heat not found")
            }

            res.end(JSON.stringify({
              message: 'New heat',
              data: heats[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/removeHeat':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  removeHeat request for:', input);

            const heats = await db.removeHeat(input.heat_id)
            if (heats.length < 1) {
              throw new Error("Heat not removed")
            }
            res.end(JSON.stringify({
              message: 'Heat removed',
              data: heats[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/addAthleteToHeat':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  addAthleteToHeat request for:', input);

            const startlist = await db.addAthleteToHeat(
                    input.athlete,
                    input.heat,
            );
            if (startlist.length < 1) {
              throw new Error("Startlist not updated")
            }

            res.end(JSON.stringify({
              message: 'Athlete added to startlist',
              data: startlist[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/removeAthleteFromHeat':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  removeAthleteFromHeat request for:', input);

            const startlist = await db.removeAthleteFromHeat(input.startlist_id)
            if (startlist.length < 1) {
              throw new Error("Startlist not updated")
            }

            res.end(JSON.stringify({
              message: 'Athlete removed from startlist',
              data: startlist[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/addAthlete':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  addAthlete request for:', input);

            const athlete = await db.addAthlete(
                    input.nr,
                    input.firstname,
                    input.lastname,
                    input.birthday,
                    input.school,
            );
            if (athlete.length < 1) {
              throw new Error("Athlete not removed")
            }

            res.end(JSON.stringify({
              message: 'Athlete created',
              data: athlete[0]
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/removeAthlete':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  removeAthlete request for:', input);

            const athlete = await db.removeAthlete(input.athlete_id)
            if (athlete.length < 1) {
              throw new Error("Athlete not removed")
            }
            res.end(JSON.stringify({
              message: 'Athlete removed',
              data: athlete[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/addJudge':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  addJudge request for:', input);

            const judge = await db.addJudge(
                    input.email,
                    input.firstname,
                    input.lastname,
            );
            if (judge.length < 1) {
              throw new Error("Judge not added")
            }

            res.end(JSON.stringify({
              message: 'Judge created',
              data: judge[0]
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/removeJudge':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  removeJudge request for:', input);

            const judge = await db.removeJudge(input.judge_id)
            if (judge.length < 1) {
              throw new Error("Judge not removed")
            }
            res.end(JSON.stringify({
              message: 'Judge removed',
              data: judge[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/updateSetting':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  updateSetting request for:', input);

            const setting = await db.updateSetting(input.name, input.value)
            if (setting.length < 1) {
              throw new Error("Setting not updated")
            }
            res.end(JSON.stringify({
              message: 'Setting updated',
              data: setting[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      case '/v1/leaderboard/removeSetting':
        req.on('data', chunk => {
          body.push(chunk);
        }).on('end', async () => {
          const b = Buffer.concat(body);
          try {
            await verifyToken(req, token)
            if (b.length < 1) {
              throw new Error("Empty request body")
            }
            input = JSON.parse(b);
            console.log('  removeSetting request for:', input);

            const settings = await db.removeSetting(input.name)
            if (settings.length < 1) {
              throw new Error("Setting not removed")
            }
            res.end(JSON.stringify({
              message: 'Setting removed',
              data: settings[0],
            }));
          } catch (error) {
            serverError(res, error);
          }
        })
        break
      default:
        const pathExists = paths.find((i) => i === url.pathname);
        if (pathExists) {
          // wrong method for this path
          notAllowed(res, req.method);
        } else {
          notFound(res, url.pathname);
        }
    } // end switch
  } else {
    notAllowed(res, req.method);
  }
});

// returns user on success, throws error otherwise
async function verifyToken(req, token) {
  if (!token){
    throw {message: "No token found in verifyToken"}
  } else if (token.length === 0) {
    throw {message: "Token empty in verifyToken"}
  }

  // lookup token in the database
  const users = await db.lookupToken(token);
  if (users.length < 1) {
    throw {message: "User not found in verifyToken"}
  }

  const user = users[0]

  console.warn("  Token expires_at:", users[0].expires_at);
  console.log("  Time now, lah:", new Date());

  // check expiration date of token in the database
  if (new Date(users[0].expires_at) < new Date()) {
    throw new Error('Token expired.. tthink about yu saad live 😿')
  }

  // verify token signature
  const v = jwt.verify(token, jwt_secret);
  //await db.invalidateToken(token);
  console.log("  Verified token", v);

  if (v.email !== user.email) {
    throw new Error("Token signature invalid")
  }
  return user
}

function noContent(res) {
  console.warn('x Warning: 204 no rice 😱');
  res.statusCode = 204;
  res.end();
}
function notFound(res, path) {
  console.error('x Error: 404 not found');
  res.statusCode = 404;
  res.end(JSON.stringify({
    message: 'no tea cup 🍵 use fingaa 👇 haaiiyaa!',
    error: `404 path ${path} not found`,
  }));
}
function notAllowed(res, method) {
  console.error('x Error: 403 method', method, 'not allowed');
  res.statusCode = 403;
  res.end(JSON.stringify({
    message: 'why english tea 🫖 cup to measure rice? Use fingaa 👇 haaiiyaa!',
    error: `403 method ${method} not allowed`,
  }));
}
function serverError(res, err) {
  console.error('x Error:', err);
  res.statusCode = 500;
  res.end(JSON.stringify({
    message: 'no colander, haaiiyaa, rice 🍚 wet you fucked up!',
    error: err.message,
  }));
}

// Create websocket listeners
// https://github.com/websockets/ws?tab=readme-ov-file#multiple-servers-sharing-a-single-https-server
const wss1 = new ws.WebSocketServer({ noServer: true});

// Listen for websocket connections
wss1.on('connection', function connection(sock) {
  sock.on('message', async function message(m) {
    try {
      msg = JSON.parse(m)
      console.log('  Uncle roger hears:', msg);

      if (msg.method === 'watchScores') {
        console.log(`~ Client requesting new scores`);
        db.watchScores(sock)
      }
    } catch (error) {
      console.error('x Error:', error.message);
    }
  });

  sock.on('error', console.error);

  // Client closes the connection
  sock.on('close', function(event) {
    console.log("  Close event:", event.code);
    console.log("  Close reason:", event.reason);
    db.removeClient(sock);
  });

  console.log(`~ Received a new websocket connection`);
  db.addClient(sock)
});

// Listen to upgrade event
server.on('upgrade', function upgrade(req, sock, head) {
  const url = new URL(`${req.protocol}://${req.host}${req.url}`);

  console.log('> WS path', url.pathname);

  if (url.pathname === '/v1/leaderboard') {
    wss1.handleUpgrade(req, sock, head, function done(sock) {
      wss1.emit('connection', sock, req);
    });
  } else {
    sock.destroy();
  }
});

// Listen and serve
server.listen(api_port);
