package main

import (
	"crypto/rand"
	"crypto/tls"
	"database/sql"
	"encoding/hex"
	"fmt"
	"github.com/emersion/go-ical"
	"github.com/jordan-wright/email"
	"github.com/julienschmidt/httprouter"
	_ "github.com/mattn/go-sqlite3"
	"github.com/pquerna/termchalk/prettytable"
	"github.com/teambition/rrule-go"
	"hivedav/caldav"
	"hivedav/config"
	"hivedav/tzdb"
	"html"
	"html/template"
	"io"
	"log"
	"net/http"
	netmail "net/mail"
	"net/smtp"
	"os"
	"regexp"
	"strconv"
	"strings"
	"time"
)

type Server struct {
	config *config.Config
	db     *sql.DB
}

type TableData struct {
	TableHead   []string
	Rows        [][]string
	Week        int
	Year        int
	Version     string
	HiveDavHost string
}

type CubicleFormData struct {
	Dtstart      string
	DtstartInput string
	Version      string
}

type IcsData struct {
	Timestamp            string
	Prodid               string
	Uid                  string
	Dtstart              string
	Dtend                string
	Timezone             string
	StandardTzOffsetFrom string
	StandardTzOffsetTo   string
	DstTzOffsetFrom      string
	DstTzOffsetTo        string
	Summary              string
	Reminder             int
	Description          string
	Location             string
	Organizer            string
	Attendee             string
}

var drop = `DROP TABLE IF EXISTS availability;
	    DROP TABLE IF EXISTS availability_1;`
var create_availability = `CREATE TABLE IF NOT EXISTS availability (
			   id INTEGER NOT NULL PRIMARY KEY,
			   start DATETIME NOT NULL,
			   end DATETIME NOT NULL,
			   recurring BOOL NOT NULL);`
var create_availability_1 = `CREATE TABLE IF NOT EXISTS availability_1 (
			     id INTEGER NOT NULL PRIMARY KEY,
			     start DATETIME NOT NULL,
			     end DATETIME NOT NULL,
			     recurring BOOL NOT NULL);`

var available = `SELECT id FROM availability WHERE
		 (start >= ? AND start < ?) OR
		 (end > ? AND end <= ?) OR
		 (start <= ? AND end >= ?);`

func (s *Server) NewServer(c *config.Config) (*Server, error) {
	s = new(Server)
	s.config = c

	// prepare the sqlite database
	db, err := sql.Open("sqlite3", "app.db")
	if err != nil {
		log.Fatal("Error opening sqlite3 database 'app.db'", err)
	}
	s.db = db

	if _, err := db.Exec(drop); err != nil {
		return nil, err
	}
	if _, err := db.Exec(create_availability); err != nil {
		return nil, err
	}
	if _, err := db.Exec(create_availability_1); err != nil {
		return nil, err
	}

	return s, nil
}

// Returns Time with requested weekday and week. If out of bounds, the returned
// week is set to the maximum for the specific year.
// https://xferion.com/golang-reverse-isoweek-get-the-date-of-the-first-day-of-iso-week
func dayOfISOWeek(weekday int, week int, year int) (timeIt time.Time, weekNow int) {
	// check out of bounds week (last week of the year)
	limit := time.Date(year, 12, 31, 0, 0, 0, 0, time.Local)
	_, limitWeek := limit.ISOWeek()
	if limitWeek == 1 {
		// last day of year already in first week of next year
		// go back a week (years, months, days)
		limit = limit.AddDate(0, 0, -7)
		_, limitWeek = limit.ISOWeek()
	}

	if week > limitWeek {
		// set week to last week of requested year
		week = limitWeek
	}

	// start time iterator in week 0
	timeIt = time.Date(year, 0, 0, 0, 0, 0, 0, time.Local)
	yearNow, weekNow := timeIt.ISOWeek()

	// iterate back to the weekday
	for timeIt.Weekday() != time.Weekday(weekday) {
		// years, months, days
		timeIt = timeIt.AddDate(0, 0, -1)
		yearNow, weekNow = timeIt.ISOWeek()
	}

	// if we iterated back into the old year, iterate forward in weekly
	// steps (7) until we reach the weekday of the first week in yearNow
	for yearNow < year {
		timeIt = timeIt.AddDate(0, 0, 7)
		yearNow, weekNow = timeIt.ISOWeek()
	}

	// iterate forward to the first day of the given ISO week
	for weekNow < week {
		timeIt = timeIt.AddDate(0, 0, 7)
		yearNow, weekNow = timeIt.ISOWeek()
	}

	return timeIt, weekNow
}

func (s *Server) available(start string, end string) (bool, error) {
	var id int

	stmt, err := s.db.Prepare(available)

	if err != nil {
		log.Println("Error during db statement Prepare()")
		return false, err
	}

	err = stmt.QueryRow(start, end, start, end, start, end).Scan(&id)
	if err != nil {
		if err == sql.ErrNoRows {
			// We are available if we cannot find any rows
			return true, nil
		}
		return false, err
	}
	return false, nil
}

var funcMap = template.FuncMap{
	"next": func(i int) int { return i + 1 },
	"prev": func(i int) int { return i - 1 },
	"mkSlice": func(a int, b int) []int {
		// return slice of integers from a to b
		var r []int
		for i := a; i <= b; i++ {
			r = append(r, i)
		}
		return r
	},
}

func (s *Server) Week(w http.ResponseWriter, req *http.Request, ps httprouter.Params) {
	userAgent := req.Header.Get("user-agent")

	// overwrite with requested year and week
	year, err := strconv.Atoi(ps.ByName("year"))
	if err != nil {
		log.Printf("Cannot serve requested year '%d'\n", year)
		http.Error(w, fmt.Sprintf("Cannot serve requested year '%d'\n", year), http.StatusBadRequest)
		return
	}
	week, err := strconv.Atoi(ps.ByName("week"))
	if err != nil {
		log.Printf("Cannot serve requested week '%d'\n", week)
		http.Error(w, fmt.Sprintf("Cannot serve requested week '%d'\n", week), http.StatusBadRequest)
		return
	}

	monday, week := dayOfISOWeek(1, week, year)
	log.Printf("Serving week '%v' of year '%v'\n", week, year)
	log.Printf("Monday is '%v'\n", monday)

	sqlDateFmt := "2006-01-02 15:04:00"

	termRows := make([][]string, 9)
	for i := range termRows {
		termRows[i] = make([]string, 6)
	}
	htmlRows := make([][]string, 9)
	for i := range htmlRows {
		htmlRows[i] = make([]string, 6)
	}

	// Define the table header
	// TODO: Make heading date format configurable, start of week, etc..
	tableHead := []string{
		"Time      ",
		fmt.Sprintf("Mon %s", monday.Format("02.01.")),
		fmt.Sprintf("Tue %s", monday.Add(time.Hour*24).Format("02.01.")),
		fmt.Sprintf("Wed %s", monday.Add(time.Hour*24*2).Format("02.01.")),
		fmt.Sprintf("Thu %s", monday.Add(time.Hour*24*3).Format("02.01.")),
		fmt.Sprintf("Fri %s", monday.Add(time.Hour*24*4).Format("02.01.")),
	}

	termTableData := TableData{
		TableHead:   tableHead,
		Rows:        termRows,
		Week:        week,
		Year:        year,
		Version:     hivedavVersion,
		HiveDavHost: s.config.HiveDavHost,
	}
	htmlTableData := TableData{
		TableHead:   tableHead,
		Rows:        htmlRows,
		Week:        week,
		Year:        year,
		Version:     hivedavVersion,
		HiveDavHost: s.config.HiveDavHost,
	}

	// TODO: use timeIt to go through the loops below, remove the static arrays
	// TODO: make display timezone configurable, display in local time
	timeIt := time.Date(monday.Year(), monday.Month(), monday.Day(), 0, 0, 0, 0, time.Local)
	//dayEnd := time.Date(monday.Year(), monday.Month(), monday.Day(), 17, 0, 0, 0, time.Local)
	//weekEnd := dayEnd.Add(time.Hour * 24*5)

	// Working hours - Eight to Five
	//for h := 8; h < 17; h++ {
	for _, h := range []int{8, 9, 10, 11, 12, 13, 14, 15, 16} {
		var htmlAvailability [5]string
		var termAvailability [5]string
		// Working days - Monday through Friday
		//for d := 0; d < 5; d++ {
		for _, d := range []int{1, 2, 3, 4, 5} {
			// convert to read/compare in UTC from db
			startTime := timeIt.Add(time.Hour * time.Duration((d-1)*24+h))
			start := startTime.UTC().Format(sqlDateFmt)
			endTime := timeIt.Add(time.Hour * time.Duration((d-1)*24+h+1))
			end := endTime.UTC().Format(sqlDateFmt)
			avi, err := s.available(start, end)
			if err != nil {
				log.Printf("Error getting availability on day '%d' hour '%d'\n", d, h)
				return
			}
			if avi {
				// terminal shows free slots as empty
				termAvailability[d-1] = ""
				// web UI shows a booking link for free slots
				htmlAvailability[d-1] = fmt.Sprintf("%s", startTime.Format("2006-01-02-15"))
			} else {
				termAvailability[d-1] = "X"
				htmlAvailability[d-1] = ""
			}
			//timeIt = timeIt.Add(time.Hour * 24)
		}
		termTableData.Rows[h-8] = []string{
			fmt.Sprintf("%02d:00 - %02d:00", h, h+1),
			termAvailability[0],
			termAvailability[1],
			termAvailability[2],
			termAvailability[3],
			termAvailability[4],
		}
		htmlTableData.Rows[h-8] = []string{
			fmt.Sprintf("%02d:00 - %02d:00", h, h+1),
			htmlAvailability[0],
			htmlAvailability[1],
			htmlAvailability[2],
			htmlAvailability[3],
			htmlAvailability[4],
		}
		//timeIt = timeIt.Add(time.Hour)
	}

	if strings.Contains(userAgent, "curl") {
		w.Header().Set("Content-Type", "text/plain")
		io.WriteString(w, fmt.Sprintf("Serving week %d of year %d\n", week, year))
		io.WriteString(w, fmt.Sprintf("List cubicle booking commands: 'curl %s/list/%d/%d'\n", s.config.HiveDavHost, year, week))
		io.WriteString(w, fmt.Sprintf("> Next week: 'curl %s/week/%d/%d'\n", s.config.HiveDavHost, year, week+1))
		io.WriteString(w, fmt.Sprintf("< Prev week: 'curl %s/week/%d/%d'\n", s.config.HiveDavHost, year, week-1))
		pt := prettytable.New(termTableData.TableHead)
		for _, r := range termTableData.Rows {
			// convert to slice of interface
			// https://go.dev/doc/faq#convert_slice_of_interface
			s := make([]interface{}, len(r))
			for i, v := range r {
				s[i] = v
			}
			// unpack the string array into row arguments
			pt.AddRow(s...)
		}

		io.WriteString(w, fmt.Sprint(pt))
		io.WriteString(w, fmt.Sprintf("HiveDAV %s 🍯\n", hivedavVersion))
	} else {
		w.Header().Set("Content-Type", "text/html")

		tmpl, err := template.New("index.html").Funcs(funcMap).ParseFiles("./templates/index.html")
		if err != nil {
			log.Printf("Error parsing html template: %v\n", err)
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
		err = tmpl.Execute(w, htmlTableData)
		if err != nil {
			log.Printf("Error executing html template: %v\n", err)
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
	}
}

func (s *Server) Index(w http.ResponseWriter, req *http.Request, _ httprouter.Params) {
	timeNow := time.Now()
	_, currentWeek := timeNow.ISOWeek()
	ps := httprouter.Params{
		httprouter.Param{"year", strconv.Itoa(timeNow.Year())},
		httprouter.Param{"week", strconv.Itoa(currentWeek)},
	}
	s.Week(w, req, ps)
}

func (s *Server) UpdateAvailability(calData []caldav.CalData) error {
	localNow := time.Now()
	isDSTNow := localNow.IsDST()
	// set limit to recurring events based on configured time horizon
	rrLimit := time.Date(localNow.Year()+s.config.Horizon, 12, 31, 0, 0, 0, 0, time.Local)

	// Walk through events and store start/end time in the database
	for _, event := range calData {
		dec := ical.NewDecoder(strings.NewReader(event.Data))
		cal, err := dec.Decode()
		var tzOffsetTo string

		if err != nil {
			return err
		}

		// check for TIMEZONE component and read TZOFFSETTO from the
		// correct sub-child
		for _, c := range cal.Children {
			if c.Name == ical.CompTimezone {
				for _, child := range c.Children {
					if isDSTNow && child.Name == ical.CompTimezoneDaylight {
						// summer time
						tzOffsetTo = child.Props.Get(ical.PropTimezoneOffsetTo).Value
					} else if !isDSTNow && child.Name == ical.CompTimezoneStandard {
						// winter time
						tzOffsetTo = child.Props.Get(ical.PropTimezoneOffsetTo).Value
					}
				}
			}
		}

		var tzOffsetDuration time.Duration
		re := regexp.MustCompile(`^([+-][0-9]{2}).*`)
		matches := re.FindStringSubmatch(tzOffsetTo)
		if len(matches) == 0 {
			//_, offset := localNow.Zone()
			//tzOffsetDuration = time.Second * time.Duration(offset)
			//tzOffsetDuration = time.Second * time.Duration(-1 * offset)
			//log.Printf("Cannot find time zone offset, using local offset: %v\n", tzOffsetDuration)
			tzOffsetDuration = time.Duration(0)
		} else {
			tzOffsetDurationInt, _ := strconv.Atoi(matches[1])
			tzOffsetDuration = time.Hour * time.Duration(tzOffsetDurationInt)
			// reset to UTC
			tzOffsetDuration = time.Hour * time.Duration(-1*tzOffsetDurationInt)
		}

		for _, e := range cal.Events() {
			startTimeProp := e.Props.Get(ical.PropDateTimeStart)
			endTimeProp := e.Props.Get(ical.PropDateTimeEnd)

			if startTimeProp == nil || endTimeProp == nil {
				// skip events with funny format, not useful anyways
				continue
			}

			start := startTimeProp.Value
			end := endTimeProp.Value

			status := e.Props.Get("X-MICROSOFT-CDO-INTENDEDSTATUS")
			statusValue := "BUSY"
			if status != nil {
				statusValue = status.Value
			}
			if statusValue == "FREE" {
				// do not store "blocker" events
				continue
			}

			// Parse all possible time formats
			// Don't use ical DateTime(), has problems with some timezones
			startTime, err := parseTime(start, tzOffsetDuration)
			if err != nil {
				return err
			}
			endTime, err := parseTime(end, tzOffsetDuration)
			if err != nil {
				return err
			}
			duration := endTime.Sub(startTime)

			// Parse the recurrence rules
			// https://github.com/emersion/go-ical/blob/master/ical.go
			// https://github.com/emersion/go-ical/blob/master/components.go
			roption, err := e.Props.RecurrenceRule()
			if err != nil {
				return err
			}

			// set new availability in temporary table availability_1
			if _, err = s.db.Exec("INSERT INTO availability_1 VALUES(NULL, DATETIME(?), DATETIME(?), false);", startTime, endTime); err != nil {
				return err
			}

			// Save recurring events
			if roption != nil {
				r, err := rrule.NewRRule(*roption)
				if err != nil {
					return err
				}

				set := rrule.Set{}
				set.RRule(r)
				set.DTStart(startTime)

				// add exception dates to recurrence set exclusion list
				exceptionDates := e.Props.Get(ical.PropExceptionDates)
				if exceptionDates != nil {
					exDateTime, err := parseTime(exceptionDates.Value, tzOffsetDuration)
					if err != nil {
						return err
					}
					set.ExDate(exDateTime)
				}

				// don't include the first event, handled outside the loop
				rDates := set.Between(startTime, rrLimit, false)
				for _, rrd := range rDates {
					if _, err = s.db.Exec("INSERT INTO availability_1 VALUES(NULL, DATETIME(?), DATETIME(?), true);", rrd, rrd.Add(duration)); err != nil {
						return err
					}
				}
			}
		}
	}

	// delete current availability
	if _, err := s.db.Exec("DROP TABLE IF EXISTS availability"); err != nil {
		return err
	}
	// set new availability as current availability
	if _, err := s.db.Exec("ALTER TABLE availability_1 RENAME TO availability"); err != nil {
		return err
	}
	// prepare new temporary table
	if _, err := s.db.Exec(create_availability_1); err != nil {
		return err
	}

	return nil
}

// parse time from local time zone and return UTC time for sqlite db
func parseTime(timeStr string, offset time.Duration) (time.Time, error) {
	t, err := time.Parse("20060102T150405Z", timeStr)
	if err != nil {
		t, err = time.Parse("20060102T150405", timeStr)
	}
	if err != nil {
		t, err = time.Parse("20060102", timeStr)
	}
	if err != nil {
		return time.Time{}, err
	}
	return t.Add(offset), nil
}

// https://github.com/kelseyhightower/app-healthz
func (s *Server) Healthz(w http.ResponseWriter, req *http.Request, _ httprouter.Params) {
	w.Header().Set("Content-Type", "application/json")
	var count = 0
	row := s.db.QueryRow("SELECT count(*) FROM availability")

	if err := row.Scan(&count); err == sql.ErrNoRows {
		log.Println("Error reading availabilty table")
	}
	if count > 0 {
		io.WriteString(w, fmt.Sprintf("{\"status\":\"wagwan, hive all good\",\"rows\":\"%d\",\"version\":\"%s\"}", count, hivedavVersion))
	} else {
		http.Error(w, fmt.Sprintf("{\"status\":\"nah fam, db not initialized\",\"rows\":\"%d\",\"version\":\"%s\"}", count, hivedavVersion), http.StatusInternalServerError)
	}
}

func (s *Server) ListCubicles(w http.ResponseWriter, req *http.Request, _ httprouter.Params) {
	timeNow := time.Now()
	_, currentWeek := timeNow.ISOWeek()
	ps := httprouter.Params{
		httprouter.Param{"week", strconv.Itoa(currentWeek)},
	}
	s.ListCubiclesInWeek(w, req, ps)
}

func (s *Server) ListCubiclesInWeek(w http.ResponseWriter, req *http.Request, ps httprouter.Params) {
	userAgent := req.Header.Get("user-agent")

	// overwrite with requested year and week
	year, err := strconv.Atoi(ps.ByName("year"))
	if err != nil {
		log.Printf("Cannot serve requested year '%d'\n", year)
		http.Error(w, fmt.Sprintf("Cannot serve requested year '%d'\n", year), http.StatusBadRequest)
		return
	}
	week, err := strconv.Atoi(ps.ByName("week"))
	if err != nil {
		log.Printf("Cannot serve requested week '%d'\n", week)
		http.Error(w, fmt.Sprintf("Cannot serve requested week '%d'\n", week), http.StatusBadRequest)
		return
	}

	monday, week := dayOfISOWeek(1, week, year)
	log.Printf("Serving week '%v' of year '%v'\n", week, year)
	log.Printf("Monday is '%v'\n", monday)

	sqlDateFmt := "2006-01-02 15:04:00"

	var rows [][]string

	// Define the table header
	// TODO: Make heading date format configurable, start of week, etc..
	tableHead := []string{
		"Date/Time             ",
		// TODO: %!v(PANIC=String method: strings: negative Repeat count)
		"Book Cubicle                                         ",
	}

	tableData := TableData{
		TableHead:   tableHead,
		Rows:        rows,
		Week:        week,
		Year:        year,
		Version:     hivedavVersion,
		HiveDavHost: s.config.HiveDavHost,
	}

	// TODO: use timeIt to go through the loops below, remove the static arrays
	timeIt := time.Date(monday.Year(), monday.Month(), monday.Day(), 0, 0, 0, 0, time.Local)

	// 5 days * 9 1hour time slots = 45 possible booking commands
	for d := 0; d < 5; d++ {
		// 9 1hour time slots
		for h := 8; h < 17; h++ {
			// convert to read/compare in UTC from db
			startTime := timeIt.Add(time.Hour * time.Duration((d)*24+h))
			endTime := timeIt.Add(time.Hour * time.Duration((d)*24+h+1))
			start := startTime.UTC().Format(sqlDateFmt)
			end := endTime.UTC().Format(sqlDateFmt)
			avi, err := s.available(start, end)
			if err != nil {
				log.Printf("Error getting availability on day '%d' hour '%d'\n", d, h)
				return
			}
			if avi {
				// Make two columns, for the date/time and the command
				tableData.Rows = append(tableData.Rows, []string{
					fmt.Sprintf("%s - %s", startTime.Format("Mon 02.01. 15:04"), endTime.Format("15:04")),
					fmt.Sprintf("%s", startTime.Format("2006-01-02-15")),
				})
			}
		}
	}

	if strings.Contains(userAgent, "curl") {
		w.Header().Set("Content-Type", "text/plain")
		io.WriteString(w, fmt.Sprintf("Serving week %d of year %d\n", week, year))
		io.WriteString(w, fmt.Sprintf("Back to calendar: 'curl %s/week/%d/%d'\n", s.config.HiveDavHost, year, week))

		pt := prettytable.New(tableData.TableHead)
		for _, r := range tableData.Rows {
			// convert to slice of interface
			// https://go.dev/doc/faq#convert_slice_of_interface
			row := make([]interface{}, len(r))

			// use the date/time column as is
			row[0] = r[0]
			// create the booking cmd
			row[1] = fmt.Sprintf("curl %s/book/%s -F 'mail=' -F 'msg='", s.config.HiveDavHost, r[1])

			// unpack the string array into row arguments
			pt.AddRow(row...)
		}

		io.WriteString(w, fmt.Sprint(pt))
		io.WriteString(w, fmt.Sprintf("HiveDAV %s 🍯\n", hivedavVersion))
	} else {
		w.Header().Set("Content-Type", "text/html")

		tmpl, err := template.New("listcubicles.html").Funcs(funcMap).ParseFiles("./templates/listcubicles.html")
		if err != nil {
			log.Printf("Error parsing html template: %v\n", err)
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
		err = tmpl.Execute(w, tableData)
		if err != nil {
			log.Printf("Error executing html template: %v\n", err)
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
	}
}

func (s *Server) CubicleForm(w http.ResponseWriter, req *http.Request, ps httprouter.Params) {
	dtstart := ps.ByName("dtstart")
	dtStartPathTimeLocal, err := time.ParseInLocation("2006-01-02-15", dtstart, time.Local)

	if err != nil {
		log.Printf("Error parsing booking dtstart: %v\n", err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	icsData := CubicleFormData{
		DtstartInput: dtStartPathTimeLocal.Format("2006-01-02T15:04"),
		Dtstart:      dtstart,
		Version:      hivedavVersion,
	}

	w.Header().Set("Content-Type", "text/html")

	tmpl, err := template.New("cubicleform.html").Funcs(funcMap).ParseFiles("./templates/cubicleform.html")
	if err != nil {
		log.Printf("Error parsing html template: %v\n", err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}
	err = tmpl.Execute(w, icsData)
	if err != nil {
		log.Printf("Error executing html template: %v\n", err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}
}

func (s *Server) BookCubicle(w http.ResponseWriter, req *http.Request, ps httprouter.Params) {
	dtStartPath := ps.ByName("dtstart")

	//TODO: replace Local TZ by config setting
	//TODO: remove Year from the input format
	dtStartPathTimeLocal, err := time.ParseInLocation("2006-01-02-15", dtStartPath, time.Local)
	if err != nil {
		log.Printf("Error parsing booking dtstart from request path: %v\n", err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}
	log.Printf("dtstart booking request (local time): %v\n", dtStartPathTimeLocal)

	// total of maxMemory bytes of its file parts are stored in memory
	// with the remainder stored on disk in temporary files
	// https://pkg.go.dev/net/http#Request.ParseMultipartForm
	err = req.ParseMultipartForm(256)
	if err != nil {
		log.Printf("ParseMultipartForm err: %v\n", err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	dtstart := dtStartPathTimeLocal
	inputDateTimeFmt := "2006-01-02T15:04"
	dtStartInput := req.FormValue("dtstart")
	if dtStartInput != "" {
		// TODO: replace local time by timezone specified in config file
		dtStartInputTime, err := time.ParseInLocation(inputDateTimeFmt, dtStartInput, time.Local)
		if err != nil {
			log.Printf("Error parsing booking dtstart from user input: %v\n", err)
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		// prefer time time chosen by the user (html input field)
		if dtStartPathTimeLocal != dtStartInputTime {
			dtstart = dtStartInputTime
			log.Printf("Preferring user input dtstart '%v' from the form\n", dtStartInput)
		}
	}

	log.Printf("dtStartInput: %v\n", dtStartInput)
	log.Printf("dtStartPath: %v\n", dtStartPath)
	log.Printf("dtstart booking request (local time): %v\n", dtstart)

	mail := req.FormValue("mail")
	//msg := req.FormValue("msg")
	msg := html.UnescapeString(req.FormValue("msg"))

	if mail == "" || msg == "" {
		errorMsg := fmt.Sprintf("Mail and msg required, try again: 'curl %s/book/%s -F 'mail=' -F 'msg='\n", s.config.HiveDavHost, dtstart.Format("2006-01-02-15"))
		http.Error(w, errorMsg, http.StatusBadRequest)
		return
	}

	// convert to read/compare in UTC from db
	sqlDateFmt := "2006-01-02 15:04:00"
	start := dtstart.UTC().Format(sqlDateFmt)
	end := dtstart.Add(time.Hour).UTC().Format(sqlDateFmt)
	avi, err := s.available(start, end)
	if err != nil {
		log.Printf("Error getting availability on '%s'\n", start)
		return
	}

	// check if the slot is still available
	if !avi {
		log.Printf("Duplicate booking detected for requestor: %s\n", mail)
		http.Error(w, fmt.Sprintf("This cubicle is already booked. Choose a free slot: 'curl %s/list'\n", s.config.HiveDavHost), http.StatusBadRequest)
		return
	}

	err = s.sendMail(mail, msg, dtstart)
	if err != nil {
		log.Printf("Error occured while sending mail: %v\n", err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	// update availability
	startTimeUtc := dtstart.UTC()
	endTimeUtc := dtstart.Add(time.Hour).UTC()
	if _, err = s.db.Exec("INSERT INTO availability VALUES(NULL, DATETIME(?), DATETIME(?), false);", startTimeUtc, endTimeUtc); err != nil {
		log.Printf("Error updating availability database with new booking: %v\n", err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	zone, _ := dtstart.Zone()
	io.WriteString(w, fmt.Sprintf("Thank you for booking on %s %s\n", dtstart.Format("02 Jan 2006 15:04"), zone))

	return
}

// book a time slot in the calendar
func (s *Server) sendMail(recepient string, msg string, dtstart time.Time) error {
	_, err := netmail.ParseAddress(recepient)
	if err != nil {
		return err
	}

	e := email.NewEmail()

	e.From = s.config.SmtpUser
	e.To = []string{
		s.config.CaldavUser,
		recepient,
	}
	e.Subject = s.config.BookingSummary
	e.Text = []byte(msg)

	// parse ics booking template
	icsTmpl, err := template.ParseFiles("templates/booking.ics")
	if err != nil {
		return err
	}

	// create ics booking from template
	tmpFilePath := fmt.Sprintf("/tmp/%s.ics", dtstart.Format("2006-01-02-15"))
	icsFile, err := os.Create(tmpFilePath)
	if err != nil {
		return err
	}
	defer icsFile.Close()
	defer os.Remove(tmpFilePath)

	uuid, _ := genUid(12)

	// get timezone and offset
	// TODO: This will probably have to change for winter (standard) time
	// TODO: Check the 1/2h timezones
	zone, offsetSec := dtstart.Zone()
	tzdbZone := tzdb.GetTzdbFromLocation(zone)
	offsetHours := offsetSec / 60 / 60

	var (
		dstTzOffsetTo        string
		dstTzOffsetFrom      string
		standardTzOffsetTo   string
		standardTzOffsetFrom string
	)

	if dtstart.IsDST() {
		dstTzOffsetTo = fmt.Sprintf("%+03d00", (offsetHours))
		standardTzOffsetFrom = fmt.Sprintf("%+03d00", (offsetHours))
		dstTzOffsetFrom = fmt.Sprintf("%+03d00", (offsetHours - 1))
		standardTzOffsetTo = fmt.Sprintf("%+03d00", (offsetHours - 1))
	} else {
		// Winter/Standard time
		standardTzOffsetTo = fmt.Sprintf("%+03d00", (offsetHours))
		dstTzOffsetFrom = fmt.Sprintf("%+03d00", (offsetHours))
		standardTzOffsetFrom = fmt.Sprintf("%+03d00", (offsetHours + 1))
		dstTzOffsetTo = fmt.Sprintf("%+03d00", (offsetHours + 1))
	}

	foldedMsg := fold(msg)

	icsData := IcsData{
		Prodid:               fmt.Sprintf("-//HiveDAV//%s//EN", hivedavVersion),
		Uid:                  strings.ToUpper(uuid),
		Timestamp:            time.Now().Format("20060102T150405Z"),
		Dtstart:              dtstart.Format("20060102T150405"),
		Dtend:                dtstart.Add(time.Hour).Format("20060102T150405"),
		Timezone:             tzdbZone,
		StandardTzOffsetFrom: standardTzOffsetFrom,
		StandardTzOffsetTo:   standardTzOffsetTo,
		DstTzOffsetFrom:      dstTzOffsetFrom,
		DstTzOffsetTo:        dstTzOffsetTo,
		Summary:              s.config.BookingSummary,
		Reminder:             s.config.BookingReminder,
		Description:          foldedMsg,
		Location:             fmt.Sprintf("%s/%s", s.config.BookingLocation, strings.ToUpper(uuid)),
		Organizer:            s.config.CaldavUser,
		Attendee:             recepient,
	}

	log.Printf("foldedMsg: %s\n", foldedMsg)
	log.Printf("Description: %s\n", icsData.Description)

	err = icsTmpl.Execute(icsFile, icsData)
	if err != nil {
		return err
	}

	e.AttachFile(tmpFilePath)
	lastSlashIndex := strings.LastIndex(tmpFilePath, "/")
	var tmpFileName string
	if lastSlashIndex != -1 {
		tmpFileName = tmpFilePath[lastSlashIndex+1:]
	}
	// https://datatracker.ietf.org/doc/html/rfc2447#section-2.4
	e.Attachments[0].ContentType = fmt.Sprintf("text/calendar; charset=utf-8; method=REQUEST; name=%s", tmpFileName)
	//e.Attachments[0].ContentType = "text/calendar; charset=utf-8; method=REQUEST"
	//fmt.Println(e.Attachments[0].ContentType)
	smtpServer := fmt.Sprintf("%s:%d", s.config.SmtpHost, s.config.SmtpPort)
	auth := smtp.PlainAuth("", s.config.SmtpUser, s.config.SmtpPassword, s.config.SmtpHost)

	if s.config.SmtpStartTls {
		tlsConfig := tls.Config{
			InsecureSkipVerify: true,
		}
		err = e.SendWithStartTLS(smtpServer, auth, &tlsConfig)
	} else {
		err = e.SendWithStartTLS(smtpServer, auth, nil)
	}

	if err != nil {
		return err
	}

	return nil
}

// simple string to rune and hex conversion in Python
// i=ord("\0"); hex(i)
func fold(s string) string {
	// convert the string unicode code points
	runeStr := []rune(s)

	// create buffer for escaped result TEXT
	buf := make([]rune, 0)

	// bufl is the current buffer size incl. \0
	var bufl = 0
	// i is the iterator in s
	var i = 0

	// escch is the char to be escaped,
	// only written when esc=true
	escch := rune(0x0) // \0
	esc := false

	for (i < len(runeStr)) || esc {
		buf = append(buf, rune(0x0))
		bufl++

		if (bufl > 1) && ((bufl % 77) == 0) {
			// break lines after 75 chars
			// split between any two characters by inserting a CRLF
			// immediately followed by a white space character
			buf[bufl-1] = rune(0xa) // newline '\n'
			escch = rune(0x20)      // whitespace ' '

			esc = true
			continue
		}

		if esc {
			// only escape char, do not advance iterator i
			buf[bufl-1] = escch
			esc = false
		} else {
			// escape characters
			// https://datatracker.ietf.org/doc/html/rfc5545#section-3.3.11
			switch runeStr[i] {
			case rune(0x5c): // backslash '\'
				buf[bufl-1] = rune(0x5c)
				escch = rune(0x5c)
				esc = true
				break
			case rune(0x3b): // semicolon ';'
				buf[bufl-1] = rune(0x5c)
				escch = rune(0x3b)
				esc = true
				break
			case rune(0x2c): // comma ','
				buf[bufl-1] = rune(0x5c)
				escch = rune(0x2c)
				esc = true
				break
			case rune(0xa): // newline '\n'
				buf[bufl-1] = rune(0x5c) // backslash '\'
				escch = rune(0x6e)       // literal 'n'
				esc = true
				break
			case rune(0xd): // carriage return '\r'
				buf[bufl-1] = rune(0x20) // whitespace ' '
				break
			default:
				// write regular character from runeStr
				buf[bufl-1] = runeStr[i]
				break
			}
			i++
		}
	}

	return string(buf)
}

// https://datatracker.ietf.org/doc/html/rfc7986#section-5.3
func genUid(n int) (string, error) {
	bytes := make([]byte, n)
	if _, err := rand.Read(bytes); err != nil {
		return "", err
	}
	return hex.EncodeToString(bytes), nil
}
