package main

import (
	"fmt"
	"github.com/julienschmidt/httprouter"
	"gopkg.in/robfig/cron.v2"
	"hivedav/caldav"
	"hivedav/config"
	"log"
	"net/http"
	"time"
)

var hivedavVersion = "v0.4"

func main() {
	// If exists, load config file from pwd
	conf := &config.Config{}
	conf, err := conf.LoadConfig(".")
	if err != nil {
		log.Fatal("Error loading config: ", err)
	}

	var server *Server
	server, err = server.NewServer(conf)
	if err != nil {
		log.Fatal("Error creating server with config: ", err)
	}

	log.Printf("HiveDAV %s, %s 🍯\n", hivedavVersion, conf.HiveDavHost)
	log.Println("----------------------------------------------------")
	log.Printf("Cron interval:\t\t%d minutes\n", conf.RefreshInterval)
	log.Printf("Calendar horizon:\t\t%d year(s)\n", conf.Horizon)
	log.Printf("CalDAV uri:\t\t\t%s\n", conf.CaldavUri)
	log.Printf("CalDAV discovery host:\t%s\n", conf.CaldavHost)
	log.Printf("CalDAV user:\t\t%s\n", conf.CaldavUser)

	c := cron.New()
	c.AddFunc(fmt.Sprintf("@every %dm", conf.RefreshInterval), func() { updateDatabase(server, c) })
	go updateDatabase(server, c)
	c.Start()

	router := httprouter.New()
	router.GET("/", server.Index)
	router.GET("/week/:year/:week", server.Week)
	router.GET("/list", server.ListCubicles)
	router.GET("/list/:year/:week", server.ListCubiclesInWeek)
	router.POST("/book/:dtstart", server.BookCubicle)
	router.GET("/book/:dtstart", server.CubicleForm)
	router.ServeFiles("/css/*filepath", http.Dir("./css"))
	router.GET("/healthz", server.Healthz)
	log.Printf("Listening:\t\t\t%s:%d\n", conf.ListenAddress, conf.ListenPort)
	log.Fatal(http.ListenAndServe(fmt.Sprintf("%s:%d", conf.ListenAddress, conf.ListenPort), router))
}

func updateDatabase(s *Server, c *cron.Cron) {
	var calData []caldav.CalData
	var calendarPath string
	var err error
	// discover calendar based on HIVEDAV_CALENDAR when no CALDAV_URI is set
	if s.config.CaldavUri == "" {
		log.Println("----------------------------------------------------")
		log.Printf("Discovering calendars on host %s...\n", s.config.CaldavHost)
		// Find current user principal
		userPrincipal, err := caldav.UserPrincipal(s.config.CaldavHost, s.config.CaldavUser, s.config.CaldavPassword)
		if err != nil {
			log.Fatal("Error reading current user principal: ", err)
		}
		log.Printf("User principal:\t\t%s\n", userPrincipal)

		// Find home set of current user principal
		homeSetPath, err := caldav.CalendarHome(s.config.CaldavHost, s.config.CaldavUser, s.config.CaldavPassword, userPrincipal, s.config.Calendar)
		if err != nil {
			log.Fatal("Error reading home set URI: ", err)
		}
		log.Printf("Calendar homeset:\t\t%s", homeSetPath)

		// Find calendar URI for the calendar specified in the config
		calendarPath, err = caldav.CalendarFromHomeSet(s.config.CaldavHost, s.config.CaldavUser, s.config.CaldavPassword, homeSetPath, s.config.Calendar)
		if err != nil {
			log.Fatal("Error reading calendar URI: ", err)
		}
		log.Printf("Calendar %d path:\t\t%s\n", s.config.Calendar, calendarPath)
	}

	log.Println("----------------------------------------------------")
	log.Println("Fetching availability from CalDAV server, please wait..")

	if s.config.CaldavUri == "" {
		// Get availability data for the calendar specified in the config
		calData, err = caldav.GetAvailability(s.config.CaldavHost, s.config.CaldavUser, s.config.CaldavPassword, calendarPath)
	} else {
		// don't perform discovery, use the CALDAV_URI as calendar url
		calData, err = caldav.GetAvailabilityByUrl(s.config.CaldavUri, s.config.CaldavUser, s.config.CaldavPassword)
	}

	if err != nil {
		log.Fatal("Error reading calendar URI: ", err)
	}

	err = s.UpdateAvailability(calData)
	if err != nil {
		log.Fatalf("Error fetching availability: %v", err)
	}

	log.Println("Database initialized")
	log.Printf("Next db refresh (cron): \t%v\n", c.Entries()[0].Schedule.Next(time.Now()).Format("2006-01-02 15:04"))
	log.Println("----------------------------------------------------")
}
