package config

import (
	"github.com/spf13/viper"
	"log"
)

type Config struct {
	HiveDavHost     string `mapstructure:"HIVEDAV_HOST"`
	ListenAddress   string `mapstructure:"HIVEDAV_LISTEN_ADDRESS"`
	ListenPort      int    `mapstructure:"HIVEDAV_LISTEN_PORT"`
	CaldavUri       string `mapstructure:"HIVEDAV_CALDAV_URI"`
	CaldavHost      string `mapstructure:"HIVEDAV_CALDAV_HOST"`
	Calendar        int    `mapstructure:"HIVEDAV_CALENDAR"`
	Horizon         int    `mapstructure:"HIVEDAV_HORIZON"`
	CaldavUser      string `mapstructure:"HIVEDAV_CALDAV_USER"`
	CaldavPassword  string `mapstructure:"HIVEDAV_CALDAV_PASSWORD"`
	RefreshInterval int    `mapstructure:"HIVEDAV_REFRESH_INTERVAL"`
	SmtpHost        string `mapstructure:"HIVEDAV_SMTP_HOST"`
	SmtpPort        int    `mapstructure:"HIVEDAV_SMTP_PORT"`
	SmtpStartTls    bool   `mapstructure:"HIVEDAV_SMTP_STARTTLS"`
	SmtpUser        string `mapstructure:"HIVEDAV_SMTP_USER"`
	SmtpPassword    string `mapstructure:"HIVEDAV_SMTP_PASSWORD"`
	BookingSummary  string `mapstructure:"HIVEDAV_BOOKING_SUMMARY"`
	BookingLocation string `mapstructure:"HIVEDAV_BOOKING_LOCATION"`
	BookingReminder int    `mapstructure:"HIVEDAV_BOOKING_REMINDER"`
}

func (c *Config) LoadConfig(path string) (*Config, error) {
	viper.AddConfigPath(path)
	// look for config file with name "app"
	viper.SetConfigName("app")
	// look for config file with name "app.env"
	viper.SetConfigType("env")

	// define some defaults
	viper.SetDefault("HIVEDAV_HOST", "hivedav.example.com")
	viper.SetDefault("HIVEDAV_LISTEN_PORT", 3737)
	viper.SetDefault("HIVEDAV_LISTEN_ADDRESS", "[::]")
	viper.SetDefault("HIVEDAV_CALENDAR", 0)
	viper.SetDefault("HIVEDAV_HORIZON", 1)
	viper.SetDefault("HIVEDAV_REFRESH_INTERVAL", 30)
	viper.SetDefault("HIVEDAV_BOOKING_SUMMARY", "HiveDAV Booking")
	viper.SetDefault("HIVEDAV_BOOKING_LOCATION", "https://meet.jit.si")
	viper.SetDefault("HIVEDAV_BOOKING_REMINDER", 15)
	viper.SetDefault("HIVEDAV_SMTP_PORT", 587)
	viper.SetDefault("HIVEDAV_SMTP_STARTTLS", true)

	err := viper.ReadInConfig()
	if err != nil {
		log.Println("app.env config file not found")
		return nil, err
	}

	err = viper.Unmarshal(c)
	if err != nil {
		log.Println("Error unmarshalling config")
		return nil, err
	}

	viper.SetEnvPrefix("HIVEDAV")
	// we either need AutomaticEnv() or BindEnv() on each var
	viper.AutomaticEnv()

	// override config file vars with vars from environment
	if viper.IsSet("HOST") {
		c.HiveDavHost = viper.GetString("HOST")
	}
	if viper.IsSet("LISTEN_ADDRESS") {
		c.ListenAddress = viper.GetString("LISTEN_ADDRESS")
	}
	if viper.IsSet("LISTEN_PORT") {
		c.ListenPort = viper.GetInt("LISTEN_PORT")
	}
	if viper.IsSet("CALDAV_URI") {
		c.CaldavUri = viper.GetString("CALDAV_URI")
	}
	if viper.IsSet("CALDAV_HOST") {
		c.CaldavHost = viper.GetString("CALDAV_HOST")
	}
	if viper.IsSet("CALENDAR") {
		c.Calendar = viper.GetInt("CALENDAR")
	}
	if viper.IsSet("HORIZON") {
		c.Horizon = viper.GetInt("HORIZON")
	}
	if viper.IsSet("CALDAV_USER") {
		c.CaldavUser = viper.GetString("CALDAV_USER")
	}
	if viper.IsSet("CALDAV_PASSWORD") {
		c.CaldavPassword = viper.GetString("CALDAV_PASSWORD")
	}
	if viper.IsSet("REFRESH_INTERVAL") {
		c.RefreshInterval = viper.GetInt("REFRESH_INTERVAL")
	}
	if viper.IsSet("SMTP_HOST") {
		c.SmtpHost = viper.GetString("SMTP_HOST")
	}
	if viper.IsSet("SMTP_PORT") {
		c.SmtpPort = viper.GetInt("SMTP_PORT")
	}
	if viper.IsSet("SMTP_STARTTLS") {
		c.SmtpStartTls = viper.GetBool("SMTP_STARTTLS")
	}
	if viper.IsSet("SMTP_USER") {
		c.SmtpUser = viper.GetString("SMTP_USER")
	}
	if viper.IsSet("SMTP_PASSWORD") {
		c.SmtpPassword = viper.GetString("SMTP_PASSWORD")
	}
	if viper.IsSet("BOOKING_SUMMARY") {
		c.BookingSummary = viper.GetString("BOOKING_SUMMARY")
	}
	if viper.IsSet("BOOKING_LOCATION") {
		c.BookingLocation = viper.GetString("BOOKING_LOCATION")
	}
	if viper.IsSet("BOOKING_REMINDER") {
		c.BookingReminder = viper.GetInt("BOOKING_REMINDER")
	}

	return c, nil
}
