package caldav

// Functions to read VEVENTs from the CalDav server:
// https://sabre.io/dav/building-a-caldav-client

import (
	"bytes"
	"encoding/xml"
	"fmt"
	"net/http"
)

type XMLUserPrincipalResponse struct {
	XMLName       xml.Name `xml:multistatus`
	UserPrincipal string   `xml:"response>propstat>prop>current-user-principal>href"`
}
type XMLHomeSetResponse struct {
	XMLName xml.Name `xml:multistatus`
	HomeSet string   `xml:"response>propstat>prop>calendar-home-set>href"`
}
type XMLHomeSetCalendarResponse struct {
	XMLName  xml.Name   `xml:multistatus`
	Response []Response `xml:"response"`
}
type Response struct {
	XMLName xml.Name `xml:response`
	Href    string   `xml:"href"`
}
type XMLCalendarResponse struct {
	XMLName xml.Name  `xml:multistatus`
	CalData []CalData `xml:"response"`
}
type CalData struct {
	XMLName xml.Name `xml:response`
	Href    string   `xml:"href"`
	Data    string   `xml:"propstat>prop>calendar-data"`
}

func UserPrincipal(caldavHost string, user string, pass string) (string, error) {
	payload := []byte(`<d:propfind xmlns:d="DAV:">
		      <d:prop>
		         <d:current-user-principal />
		      </d:prop>
		    </d:propfind>`)
	reader := bytes.NewReader(payload)
	req, err := http.NewRequest("PROPFIND", caldavHost, reader)
	if err != nil {
		return "", err
	}
	req.Header.Set("Depth", "0")
	req.Header.Set("Content-Type", "text/xml")
	req.SetBasicAuth(user, pass)

	client := &http.Client{
		CheckRedirect: func(req *http.Request, via []*http.Request) error {
			// return real redirect response code (>=300)
			return http.ErrUseLastResponse
		},
	}
	res, err := client.Do(req)
	if err != nil {
		return "", err
	}
	defer res.Body.Close()

	redirectUri, err := res.Location()

	if res.StatusCode >= 300 && res.StatusCode <= 399 {
		// follow first redirect to suggested uri, repeat request
		// some providers are not discoverable starting at caldavHost
		req, err = http.NewRequest("PROPFIND", caldavHost+redirectUri.Path, reader)
		req.Header.Set("Depth", "0")
		req.Header.Set("Content-Type", "text/xml")
		req.SetBasicAuth(user, pass)
		res, err = client.Do(req)
		if err != nil {
			return "", err
		}
	}

	if res.StatusCode != 207 {
		return "", err
	}

	decoder := xml.NewDecoder(res.Body)
	var xmlResponse XMLUserPrincipalResponse
	err = decoder.Decode(&xmlResponse)
	if err != nil {
		return "", err
	}

	return xmlResponse.UserPrincipal, nil
}

func CalendarHome(caldavHost string, user string, pass string, userPrincipal string, calendar int) (string, error) {
	payload := []byte(`<d:propfind xmlns:d="DAV:" xmlns:c="urn:ietf:params:xml:ns:caldav">
			     <d:prop>
				<c:calendar-home-set />
			     </d:prop>
			   </d:propfind>`)
	reader := bytes.NewReader(payload)
	principalPath := fmt.Sprintf("%s%s", caldavHost, userPrincipal)

	req, err := http.NewRequest("PROPFIND", principalPath, reader)
	if err != nil {
		return "", err
	}
	req.Header.Set("Depth", "0")
	req.Header.Set("Content-Type", "text/xml")
	req.SetBasicAuth(user, pass)

	res, err := http.DefaultClient.Do(req)
	defer res.Body.Close()
	if err != nil {
		return "", err
	}

	if res.StatusCode != 207 {
		return "", err
	}

	decoder := xml.NewDecoder(res.Body)
	var xmlResponse XMLHomeSetResponse
	err = decoder.Decode(&xmlResponse)
	if err != nil {
		return "", err
	}

	return xmlResponse.HomeSet, nil
}

func CalendarFromHomeSet(caldavHost string, user string, pass string, homeSetPath string, calendar int) (calendarPath string, err error) {
	payload := []byte(`<d:propfind xmlns:d="DAV:" xmlns:cs="http://calendarserver.org/ns/" xmlns:c="urn:ietf:params:xml:ns:caldav">
			     <d:prop>
			        <d:resourcetype />
			        <d:displayname />
			        <cs:getctag />
			        <c:supported-calendar-component-set />
			     </d:prop>
			   </d:propfind>`)
	reader := bytes.NewReader(payload)
	homeSetUrl := fmt.Sprintf("%s%s", caldavHost, homeSetPath)

	req, err := http.NewRequest("PROPFIND", homeSetUrl, reader)
	if err != nil {
		return "", err
	}
	req.Header.Set("Depth", "1")
	req.Header.Set("Content-Type", "text/xml")
	req.SetBasicAuth(user, pass)

	res, err := http.DefaultClient.Do(req)
	defer res.Body.Close()
	if err != nil {
		return "", err
	}

	if res.StatusCode != 207 {
		return "", err
	}

	decoder := xml.NewDecoder(res.Body)
	var xmlResponse XMLHomeSetCalendarResponse
	err = decoder.Decode(&xmlResponse)
	if err != nil {
		return "", err
	}

	defer func() {
		if r := recover(); r != nil {
			err = fmt.Errorf("Calendar '%d' not found in homeset: %v", calendar, r)
		}
	}()

	calendarPath = xmlResponse.Response[calendar].Href
	return calendarPath, nil
}

func GetAvailability(caldavHost string, user string, pass string, calendarPath string) (data []CalData, err error) {
	// concatenant the host and the calendar path
	calendarUrl := fmt.Sprintf("%s%s", caldavHost, calendarPath)
	return GetAvailabilityByUrl(calendarUrl, user, pass)
}

func GetAvailabilityByUrl(calendarUrl string, user string, pass string) (data []CalData, err error) {
	payload := []byte(`<c:calendar-query xmlns:d="DAV:" xmlns:c="urn:ietf:params:xml:ns:caldav">
			       <d:prop>
			           <d:getetag />
			           <c:calendar-data />
			       </d:prop>
			       <c:filter>
			           <c:comp-filter name="VCALENDAR">
			               <c:comp-filter name="VEVENT" />
			           </c:comp-filter>
			       </c:filter>
			   </c:calendar-query>`)
	reader := bytes.NewReader(payload)

	req, err := http.NewRequest("REPORT", calendarUrl, reader)
	if err != nil {
		return data, err
	}
	req.Header.Set("Depth", "1")
	req.Header.Set("Content-Type", "text/xml")
	req.SetBasicAuth(user, pass)

	res, err := http.DefaultClient.Do(req)
	defer res.Body.Close()
	if err != nil {
		return data, err
	}

	if res.StatusCode != 207 {
		return data, err
	}

	decoder := xml.NewDecoder(res.Body)
	var xmlResponse XMLCalendarResponse
	err = decoder.Decode(&xmlResponse)
	if err != nil {
		return data, err
	}

	data = xmlResponse.CalData
	return data, nil
}
