# HiveDAV

A curlable free/busy scheduler with CalDAV integration.

## Running the Server
```bash
go run .
```

## Usage
For example, to get the availability in the next week:
```
curl http://127.0.0.1:3737/week/$(date +%Y)/$(($(date +%W)+1))
```

To query the availability in the current week:
```
curl http://127.0.0.1:3737
```

## Building
```bash
make
```

* The [sqlite3 driver](https://github.com/mattn/go-sqlite3) requires gcc and `CGO_ENABLED=1`

## Linting
```bash
make fmt
```

## Testing
Run tests:
```bash
make test
```

Generate test coverage report:
```bash
make test/cover
```

## Configuration
The application is configurable through environment variables or the
`./app.env` file:

```bash
# Optional
#HIVEDAV_LISTEN_ADDRESS=127.0.0.1
#HIVEDAV_LISTEN_PORT=3737
# only useful for discovery of the calendar when HIVEDAV_CALDAV_HOST is set
#HIVEDAV_CALENDAR=0
#HIVEDAV_HORIZON=1
#HIVEDAV_BOOKING_SUBJ="HiveDAV Booking"
#HIVEDAV_BOOKING_REMINDER=15
#HIVEDAV_SMTP_PORT=587
#HIVEDAV_SMTP_STARTTLS=true

# Required
HIVEDAV_HOST=hivedav.example.com
# an absolute uri to a caldav calendar, mutually exclusive with HIVEDAV_CALDAV_HOST
HIVEDAV_CALDAV_URI=
# the host for caldav discovery might have a port, mutually exclusive with HIVEDAV_CALDAV_URI
HIVEDAV_CALDAV_HOST=
HIVEDAV_CALDAV_USER=
# make sure to quote in single quotes to properly esacpe special chars
HIVEDAV_CALDAV_PASSWORD=''
HIVEDAV_REFRESH_INTERVAL=30
HIVEDAV_SMTP_HOST=
HIVEDAV_SMTP_USER=
# make sure to quote in single quotes to properly esacpe special chars
HIVEDAV_SMTP_PASSWORD=''
```

There is an example config file provided in `./app.env.example`.

* `HIVEDAV_CALDAV_URI` is an absolute uri to a calendar resource on a caldav
  server. No discovery is performed. You either have to define
  `HIVEDAV_CALDAV_URI` or `HIVEDAV_CALDAV_HOST`.
* With `HIVEDAV_CALDAV_HOST`, HiveDAV performs a discovery of the calendar uri
  using `HIVEDAV_CALENDAR`.
* The `HIVEDAV_LISTEN_*` variables specify the http server `ADDRESS` and `PORT`
  (bind address and port)
* The `HIVEDAV_CALENDAR` is a number that points to the index of the calendar
  collection in the ["calendar home
set"](https://www.rfc-editor.org/rfc/rfc4791.html#section-6.2.1) which should
be used to read availability data
* The `HIVEDAV_HORIZON` (number in years) limits recurring events without end
  time. 0 means that only recurring events in the current year are shown.
Unique events are not affected by `HIVEDAV_HORIZON`.
* `HIVEDAV_BOOKING_SUBJ` is the subject of the calendar invite for new booking
  requests
* `HIVEDAV_BOOKING_REMINDER` is the reminder time in minutes before the start
  of the calendar appointment
* `HIVEDAV_HOST` is the fqdn of the HiveDAV service. It is used to print the
  help commands in the curlable interface.
* `HIVEDAV_CALDAV_*` are settings to connect to the CalDAV server for fetching
  the avialability data
* TODO: specify calendar using the "Name" propery
* `HIVEDAV_REFRESH_INTERVAL` is the time in minutes between the automatic
  refresh of the database with the latest availability data from the CalDAV
server. A [cron job](https://pkg.go.dev/github.com/robfig/cron) automatically
downloads the latest events from the CalDAV server after that time. Defaults to
30 minutes.
* The `HIVEDAV_SMTP_*` settings are required to send out the ical invites for
  new appointments.

## Storage
The application stores the event start/end times in an sqlite database in the
directory where the process is running.

The schema of the `availability` table is very simple and straight forward:

```bash
sqlite3 app.db ".schema availability"
var create_availability = `CREATE TABLE IF NOT EXISTS availability (
			   id INTEGER NOT NULL PRIMARY KEY,
			   start DATETIME NOT NULL,
			   end DATETIME NOT NULL,
			   recurring BOOL NOT NULL);`
```

To show the current availability time slots:
```bash
sqlite3 app.db "select * from availability;" | less
sqlite3 app.db "select * from availability where start between '2023-01-01' and '2023-02-01';" | less
```

There exists another table `availability_1` which is used to store the new
availability time slots during an availability update ("CalDAV request").
During this CalDAV request, the table `availability_1` is used to temporarily
store the new availability. After the update is done, the temporary table is
set as the "current" availability. The application only reads availability from
the `availability` table, not from the temporary `availability_1` table.

## CalDAV Requests
The CalDAV requests in the `caldav` module are used to query:
* the user principal
* the calendar home set
* the calendar from the home set (`HIVEDAV_CALENDAR`) used as source of
  availability data

The CalDAV requests are mad according to the examples in
https://sabre.io/dav/building-a-caldav-client.

The requests are kept very simple intentionally with libraries included in Go:
* The requests only use the `net/http` library to do the HTTP requests.
* The response is parsed using the `encoding/xml` library

No external library was used perform the requests, because these libraries
typically assume specific behavior for the remote CalDAV server. Unfortunately,
the servers vary in their behavior quite a lot. For instance, certain servers
don't support the [`calendar-query
REPORT`](https://www.rfc-editor.org/rfc/rfc4791.html#section-7.8). To make this
application work with as many remote servers as possible, all events from the
remote calendar are fetched during an update and the availability data is
stored in a local database for further processing.

For more information refer to [`./docs/CALDAV.md`](./docs/CALDAV.md)

## Availability Query

- s: start time
- e: end time
- ?: sql query parameter (start/end)
- |: time slot (1h)

The (un)availability query returns true if an existing appointment is found
(unavailable, no free time slot). It returns false, when there exists no
existing calendar entry during the time slot (available).

```
//  e ?   ?
//  - |   |   false
//
//  s e   ?
//  - +   |   false
//
//  s ? e ?
//  - | - |   true (2)
//
//  s ?   e
//  - |   +   true (2)
//
//  s ?   ? e
//  - |   | - true (3)
//
//    s e ?
//    + - |   true (1)
//
//    s   e
//    +   +   true (3)
//
//    s   ? e
//    +   | - true (1)
//
//    ? s e
//    | - +   true (1)
//
//    ? s ? e
//    | - | - true (1)
//
//    ?   s e
//    |   + - false
```

Note:
The calendar does not show events marked with the
(X-MICROSOFT-CDO-INTENDEDSTATUS") "FREE" status. This can be helpful to skip
certain "reminders" in your calendar, which should not be considered as
"blocked time" (e.g., full day events).

TODO:
- Time slots are 1h each (TODO: make it configurable)
- Picture for the query

## Development & Contributions
* All source code is available in this repository. Contributions are always
  welcome! Pull requests at https://pr.in0rdr.ch/repos/hivedav
* A [Kanban
  board](https://board.in0rdr.ch/public/board/01171363fae4b8b81af1579708f7cda6d389062f05b75db860621eaed32b)
documents plans, todos and decisions for further development of the project.

## Community and Support
For question and support visit
[#p0c/libera](https://web.libera.chat/gamja/#p0c) on IRC.
